import React, {ReactNode} from 'react';
import {Helmet} from 'react-helmet';
import _isEmpty from 'lodash/isEmpty';

import {createUniqueReactKeyFromObject} from './utilities/createUniqueReactKeyFromObject';

type TOpenGraphTagAttributes = {
    content: string;
    property: string;
};

type TTwitterTagAttributes = {
    content: string;
    name: string;
};

type TSocialTagAttributes = TOpenGraphTagAttributes | TTwitterTagAttributes;

/**
 * @see https://ogp.me/
 * https://developer.twitter.com/en/docs/tweets/optimize-with-cards/overview/summary-card-with-large-image
 * https://developers.facebook.com/docs/sharing/webmasters/
 **/
export interface ISocialSharingMetaProps {
    /** "Канонический URL", используемый в качестве перманентного ID для кеширования медиа объекта (карточки) */
    /** Играет роль ID-шника при кешировании информации скраперами */
    url: string;
    title?: string;
    description?: string;
    /** Ссылка на картинку PNG | JPEG | WEBP */
    image?: string;
    imageWidth?: number | string;
    imageHeight?: number | string;
    /** Для картинок размерами больше или равно 300x157 можно использовать 'summary_large_image' */
    twitterCardType?: 'summary' | 'summary_large_image';
    /** @example "website" */
    type?: string;
    /** @example "ru_RU" */
    locale?: string;
    /** Массив атрибутов для дополнительных метатегов (будут прокинуты в качестве атрибутов метатега) */
    extra?: TSocialTagAttributes[];
}

/**
 * Компонент для единообразной работы с <meta>-тегами для СОЦСЕТЕЙ.
 * @see {DocumentMeta} для более общих мета-тегов
 * @uses {react-helmet}
 **/
class SocialSharingMeta extends React.Component<ISocialSharingMetaProps> {
    private static prepareOpenGraphTagAttributes(
        name: string,
        value: string,
    ): TOpenGraphTagAttributes {
        return {property: name, content: value};
    }

    private static prepareTwitterTagAttributes(
        name: string,
        value: string,
    ): TTwitterTagAttributes {
        return {name: name, content: value};
    }

    private static renderMeta(
        metaAttrs: TSocialTagAttributes,
    ): React.ReactNode {
        return (
            <meta
                key={SocialSharingMeta.getUniqueKey(metaAttrs)}
                {...metaAttrs}
            />
        );
    }

    private static getUniqueKey(metaAttrs: TSocialTagAttributes): string {
        return createUniqueReactKeyFromObject(metaAttrs);
    }

    render(): ReactNode {
        const {
            url,
            title,
            description,
            image,
            imageWidth,
            imageHeight,
            twitterCardType = 'summary',
            type,
            locale,
            extra,
        } = this.props;

        const meta = [
            SocialSharingMeta.prepareTwitterTagAttributes(
                'twitter:card',
                twitterCardType,
            ),
            title &&
                SocialSharingMeta.prepareTwitterTagAttributes(
                    'twitter:title',
                    title,
                ),
            description &&
                SocialSharingMeta.prepareTwitterTagAttributes(
                    'twitter:description',
                    description,
                ),
            image &&
                SocialSharingMeta.prepareTwitterTagAttributes(
                    'twitter:image',
                    image,
                ),

            type &&
                SocialSharingMeta.prepareOpenGraphTagAttributes(
                    'og:type',
                    type,
                ),
            url &&
                SocialSharingMeta.prepareOpenGraphTagAttributes('og:url', url),
            title &&
                SocialSharingMeta.prepareOpenGraphTagAttributes(
                    'og:title',
                    title,
                ),
            description &&
                SocialSharingMeta.prepareOpenGraphTagAttributes(
                    'og:description',
                    description,
                ),
            image &&
                SocialSharingMeta.prepareOpenGraphTagAttributes(
                    'og:image',
                    image,
                ),
            imageWidth !== undefined &&
                SocialSharingMeta.prepareOpenGraphTagAttributes(
                    'og:image:width',
                    String(imageWidth),
                ),
            imageHeight !== undefined &&
                SocialSharingMeta.prepareOpenGraphTagAttributes(
                    'og:image:height',
                    String(imageHeight),
                ),
            locale &&
                SocialSharingMeta.prepareOpenGraphTagAttributes(
                    'og:locale',
                    locale,
                ),
        ]
            .concat(extra || [])
            .filter((x): x is TSocialTagAttributes => Boolean(x));

        return (
            <Helmet>
                {!_isEmpty(meta) && meta.map(SocialSharingMeta.renderMeta)}
            </Helmet>
        );
    }
}

export default SocialSharingMeta;
