import React from 'react';

import {ESortType} from 'types/common/sort/ESortType';
import {ESortDirection} from 'types/common/sort/ESortDirection';
import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import SortBy from './components/SortBy/SortBy';
import Flex from 'components/Flex/Flex';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';

import cx from './SortDesktop.scss';

interface ISortDesktopProps extends IWithClassName, IWithQaAttributes {
    sortOrder?: ESortType[];
    type: ESortType;
    direction: ESortDirection;
    loading?: boolean;
    onClick(sortType: ESortType, sortDirection: ESortDirection): void;
}

export const SORTS_ORDER = [
    ESortType.DEPARTURE,
    ESortType.ARRIVAL,
    ESortType.PRICE,
    ESortType.DURATION,
];

const SortDesktop: React.FC<ISortDesktopProps> = props => {
    const {
        className,
        sortOrder = SORTS_ORDER,
        type,
        direction,
        loading = false,
        onClick,
    } = props;

    return (
        <Flex
            className={className}
            alignItems={'baseline'}
            between={5}
            inline
            {...prepareQaAttributes(props)}
        >
            {loading ? (
                <TextSkeleton
                    className={cx('skeleton')}
                    size="s"
                    withAnimation
                />
            ) : (
                sortOrder.map(sortType => (
                    <SortBy
                        key={sortType}
                        type={sortType}
                        direction={direction}
                        isActive={sortType === type}
                        onClick={onClick}
                        {...prepareQaAttributes({
                            parent: props,
                            key: sortType,
                        })}
                    />
                ))
            )}
        </Flex>
    );
};

export default SortDesktop;
