import React, {useCallback, useLayoutEffect, useRef, useState} from 'react';

import {IWithClassName} from 'types/withClassName';

import {checkNativeStickySupport} from 'utilities/checkNativeStickySupport';

interface IStickyColumnProps extends IWithClassName {
    children: React.ReactNode;
    /**
     * sticky-отступ от верха, должен быть константным
     */
    offsetTop?: number;
    /**
     * sticky-отступ от низа, должен быть константным
     */
    offsetBottom?: number;
}

/**
 * Режим, в котором находится компонент:
 *
 * STICKY_TOP: компонент приклеен к верху
 * STICKY_BOTTOM: комонент приклеен к низу
 * SCROLL: компонент не приклеен и находится в режиме скролла вместе с внешним контентом
 */
enum EStickyMode {
    STICKY_TOP = 'STICKY_TOP',
    STICKY_BOTTOM = 'STICKY_BOTTOM',
    SCROLL = 'SCROLL',
}

/**
 * Компонент скроллящегося sticky-столбца, который прилипает к низу, когда при скролле вниз
 * его контент доскроллился, а внешний - нет. Аналогично он прилипает к верху, когда
 * при скролле вверх его контент доскроллился, а внешний еще нет
 *
 * Компонент может не работать в произвольном случае, возможно его придется дорабатывать под конкретный кейс
 * в случае его переиспользования. В данный момент есть несколько ограничений:
 *
 * - столбец должен быть дочерним элементом флексбокса (используется align-self для sticky)
 * - считается, что скроллящимся элементом является window
 * - считается, что у родителя нет паддингов
 */
const StickyColumn: React.FC<IStickyColumnProps> = props => {
    const {className, children, offsetTop = 0, offsetBottom = 0} = props;

    const [rootStyle, setRootStyle] = useState<React.CSSProperties>({
        position: 'sticky',
        top: 0,
        alignSelf: 'flex-start',
    });

    const rootRef = useRef<HTMLDivElement | null>(null);
    const modeRef = useRef(EStickyMode.STICKY_TOP);
    const lastScrollYRef = useRef(
        typeof window === 'undefined' ? 0 : window.scrollY,
    );
    /**
     * Текущая высота столбца (с учетом скролла)
     */
    const nodeHeightRef = useRef(0);
    /**
     * Текущая высота родителя
     */
    const parentHeightRef = useRef(0);
    /**
     * Текущий верхний отступ от родителя
     */
    const nodeOffsetTopRef = useRef(0);
    /**
     * Отступ родителя от верха окна
     */
    const parentOffsetTopRef = useRef(0);

    /**
     * Вычисляет, есть ли скролл у столбца
     */
    const isNoNodeScroll = useCallback(() => {
        const {innerHeight: windowHeight} = window;
        const nodeHeight = nodeHeightRef.current;

        return offsetTop + offsetBottom <= windowHeight - nodeHeight;
    }, [offsetBottom, offsetTop]);

    /**
     * Вычисляет отступ от родителя при условии прикрепления сверху
     */
    const getOffsetTopForStickyTop = useCallback(() => {
        const {scrollY} = window;
        const parentOffsetTop = parentOffsetTopRef.current;

        return Math.max(
            offsetTop - parentOffsetTop,
            scrollY - parentOffsetTop + offsetTop,
        );
    }, [offsetTop]);

    /**
     * Вычисляет отступ от родителя при условии прикрепления снизу
     */
    const getOffsetTopForStickyBottom = useCallback(() => {
        const {scrollY, innerHeight: windowHeight} = window;
        const parentHeight = parentHeightRef.current;
        const parentOffsetTop = parentOffsetTopRef.current;
        const nodeHeight = nodeHeightRef.current;

        return Math.min(
            parentHeight - offsetBottom - nodeHeight,
            scrollY -
                parentOffsetTop -
                offsetBottom +
                windowHeight -
                nodeHeight,
        );
    }, [offsetBottom]);

    const changeMode = useCallback(
        (mode: EStickyMode, force?: boolean) => {
            if (mode === modeRef.current && !force) {
                return;
            }

            modeRef.current = mode;

            if (mode === EStickyMode.SCROLL) {
                setRootStyle({
                    position: 'relative',
                    top: nodeOffsetTopRef.current,
                    alignSelf: 'flex-start',
                });
            } else if (mode === EStickyMode.STICKY_TOP) {
                setRootStyle({
                    position: 'sticky',
                    top: offsetTop,
                    alignSelf: 'flex-start',
                });
            } else {
                setRootStyle({
                    position: 'sticky',
                    bottom: offsetBottom,
                    alignSelf: 'flex-end',
                });
            }
        },
        [offsetBottom, offsetTop],
    );

    /**
     * Запускается на каждый скролл и ресайз окна
     */
    const stickContainer = useCallback(() => {
        const root = rootRef.current;

        if (!root) {
            return;
        }

        const {parentElement} = root;

        if (!parentElement) {
            return;
        }

        const {scrollY} = window;
        const prevScrollY = lastScrollYRef.current;
        const scrollDiff = scrollY - prevScrollY;
        const nodeOffsetTop = nodeOffsetTopRef.current;

        lastScrollYRef.current = scrollY;

        // Высота столбца слишком маленькая, скролла нет - приклеиваем к верху
        if (isNoNodeScroll()) {
            changeMode(EStickyMode.STICKY_TOP);
        } else if (scrollDiff >= 0) {
            const currentNodeOffsetTop = getOffsetTopForStickyBottom();

            // Контент проскроллен вниз, нижняя граница столбца располагается выше нижней границы окна
            if (nodeOffsetTop <= currentNodeOffsetTop) {
                nodeOffsetTopRef.current = currentNodeOffsetTop;

                changeMode(EStickyMode.STICKY_BOTTOM);
            } else {
                changeMode(EStickyMode.SCROLL);
            }
        } else {
            const currentNodeOffsetTop = getOffsetTopForStickyTop();

            // Контент проскроллен вверх, верхняя граница столбца располагается выше верхней границы окна
            if (nodeOffsetTop >= currentNodeOffsetTop) {
                nodeOffsetTopRef.current = currentNodeOffsetTop;

                changeMode(EStickyMode.STICKY_TOP);
            } else {
                changeMode(EStickyMode.SCROLL);
            }
        }
    }, [
        changeMode,
        getOffsetTopForStickyBottom,
        getOffsetTopForStickyTop,
        isNoNodeScroll,
    ]);

    const windowResizeListener = useCallback(() => {
        stickContainer();
    }, [stickContainer]);

    const scrollListener = useCallback(() => {
        if (window.scrollY !== lastScrollYRef.current) {
            stickContainer();
        }

        // https://st.yandex-team.ru/TRAVELFRONT-5595#60e4dc40f5c1e278be08176c
        // Костыль для firefox: баг скорее всего вызван из-за вложенного sticky внутри другого sticky
        // и специфичен для firefox. Костыль состоит в форсированном ререндере контента после скролла
        if (modeRef.current === EStickyMode.STICKY_TOP) {
            const root = rootRef.current;

            if (!root) {
                return;
            }

            const position = root.style.position;

            root.style.position = 'relative';
            root.getBoundingClientRect();
            root.style.position = position;
        }
    }, [stickContainer]);

    /**
     * Запускается на ресайз элемента, возможны следующие случаи:
     *
     * --- высота столбца уменьшилась - тогда если его нижняя граница
     *     становится выше нижней границы окна, надо прилепить к низу
     * --- высота столбца увеличилась - тогда надо проконтроллировать, чтобы
     *     элемент не вывалился за пределы родителя
     */
    const nodeResizeListener = useCallback(() => {
        const root = rootRef.current;

        if (!root) {
            return;
        }

        const prevNodeHeight = nodeHeightRef.current;
        const {height: nodeHeight} = root.getBoundingClientRect();

        if (prevNodeHeight === nodeHeight) {
            return;
        }

        nodeHeightRef.current = nodeHeight;

        const parentHeight = parentHeightRef.current;
        const mode = modeRef.current;
        const nodeOffsetTop = nodeOffsetTopRef.current;

        // Высота столбца стала слишком маленькая, скролла нет - приклеиваем к верху
        if (isNoNodeScroll()) {
            changeMode(EStickyMode.STICKY_TOP);
        } else if (nodeHeight > prevNodeHeight) {
            // Высота столбца увеличилась, похоже на случай скролла вверх, когда низ блока уезжает вниз

            // Простое расширение столбца вниз приведет к тому, что он "вывалится" за родителя
            // Надо отодвинуть частично вверх, чтобы избежать этого; превращаем в скролл в любом случае
            if (nodeOffsetTop + nodeHeight > parentHeight - offsetBottom) {
                nodeOffsetTopRef.current = Math.max(
                    0,
                    parentHeight - offsetBottom - nodeHeight,
                );

                changeMode(EStickyMode.SCROLL, true);
            } else if (mode === EStickyMode.STICKY_BOTTOM) {
                // Нод расширился вниз; если он был прилипшим к низу, то он стал скроллиться

                changeMode(EStickyMode.SCROLL);
            }
        } else {
            // Высота столбца уменьшилась, похоже на случай скролла вниз, когда низ блока уезжает вверх

            const currentNodeOffsetTop = getOffsetTopForStickyBottom();

            // Контент проскроллен вниз, нижняя граница столбца располагается выше нижней границы окна
            if (nodeOffsetTop <= currentNodeOffsetTop) {
                nodeOffsetTopRef.current = currentNodeOffsetTop;

                changeMode(EStickyMode.STICKY_BOTTOM);
            }
        }
    }, [changeMode, getOffsetTopForStickyBottom, isNoNodeScroll, offsetBottom]);

    /**
     * Запускается на ресайз родителя, тут только один особый случай:
     *
     * --- высота родителя уменьшилась и теперь столбец мог вылезти за его пределы
     */
    const parentResizeListener = useCallback(() => {
        const root = rootRef.current;
        const parentElement = root?.parentElement;

        if (!parentElement) {
            return;
        }

        const nodeHeight = nodeHeightRef.current;
        const {height: parentHeight} = parentElement.getBoundingClientRect();
        const nodeOffsetTop = nodeOffsetTopRef.current;

        // Высота родителя уменьшилась, столбец мог "вылезти" за его пределы
        if (nodeOffsetTop + nodeHeight > parentHeight) {
            nodeOffsetTopRef.current = Math.max(0, parentHeight - nodeHeight);

            changeMode(EStickyMode.SCROLL, true);
        }

        parentHeightRef.current = parentHeight;
    }, [changeMode]);

    useLayoutEffect(() => {
        if (!checkNativeStickySupport()) {
            return;
        }

        const root = rootRef.current;
        const parentElement = root?.parentElement;

        if (!root || !parentElement) {
            return;
        }

        const {height} = root.getBoundingClientRect();
        const {height: parentHeight, top: parentTop} =
            parentElement.getBoundingClientRect();

        nodeHeightRef.current = height;
        parentHeightRef.current = parentHeight;

        parentOffsetTopRef.current = parentTop + scrollY;

        stickContainer();

        window.addEventListener('resize', windowResizeListener);
        window.addEventListener('scroll', scrollListener);

        const nodeResizeObserver = new ResizeObserver(nodeResizeListener);
        const parentResizeObserver = new ResizeObserver(parentResizeListener);

        nodeResizeObserver.observe(root);
        parentResizeObserver.observe(parentElement);

        return (): void => {
            window.removeEventListener('resize', windowResizeListener);
            window.removeEventListener('scroll', scrollListener);

            nodeResizeObserver.disconnect();
            parentResizeObserver.disconnect();
        };
    }, [
        nodeResizeListener,
        parentResizeListener,
        scrollListener,
        stickContainer,
        windowResizeListener,
    ]);

    return (
        <div ref={rootRef} className={className} style={rootStyle}>
            {children}
        </div>
    );
};

export default React.memo(StickyColumn);
