import React, {useCallback} from 'react';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18nNew/components-SubscriptionModal';

import Text from 'components/Text/Text';
import Heading from 'components/Heading/Heading';
import ModalOrBottomSheet from 'components/ModalOrBottomSheet/ModalOrBottomSheet';
import SubscriptionForm, {
    ISubscriptionFormProps,
} from 'components/SubscriptionForm/SubscriptionForm';

import subscriptionsMobile from 'images/subscriptionsMobile.png';
import subscriptionsDesktop from 'images/subscriptionsDesktop.png';

import cx from './SubscriptionModal.scss';

interface ISubscriptionModalProps extends ISubscriptionFormProps {
    title?: React.ReactNode;
    text?: React.ReactNode;
    submitButton?: React.ReactNode;
    disclaimer?: React.ReactNode;
    inputPlaceholder?: string;
    email?: string;
    isVisible: boolean;

    onClose(): void;
    onSubmit(email: string): void | Promise<void>;
}

const SubscriptionModal: React.FC<ISubscriptionModalProps> = props => {
    const {
        title = i18nBlock.title(),
        text = i18nBlock.text(),
        disclaimer = i18nBlock.disclaimer(),
        isVisible,
        onSubmit,
        onClose,
        ...rest
    } = props;

    const deviceType = useDeviceType();

    const handleSubscribe = useCallback(
        async (email: string) => {
            await onSubmit(email);

            setTimeout(onClose, 500);
        },
        [onSubmit, onClose],
    );

    const wrapperClassName = cx('root', deviceMods('root', deviceType));

    return (
        <ModalOrBottomSheet
            isVisible={isVisible}
            onClose={onClose}
            modalProps={{
                preventBodyScroll: false,
                isMobile: deviceType.isMobile,
                hasCloseButton: true,
            }}
            modalContentProps={{className: wrapperClassName}}
        >
            <div className={deviceType.isMobile ? wrapperClassName : undefined}>
                <img
                    className={cx('image')}
                    src={
                        deviceType.isMobile
                            ? subscriptionsMobile
                            : subscriptionsDesktop
                    }
                    alt={i18nBlock.imageAlt()}
                />
                <div className={cx('body')}>
                    <Heading level={2} className={cx('title')}>
                        {title}
                    </Heading>

                    {Boolean(text) && (
                        <Text className={cx('text')} size="m">
                            {text}
                        </Text>
                    )}

                    <SubscriptionForm
                        className={cx('form')}
                        inputClassName={cx('input')}
                        onSubmit={handleSubscribe}
                        restoreDefaultStateTimeout={1000}
                        {...rest}
                    />
                    <Text
                        className={cx('disclaimer')}
                        size="s"
                        color="secondary"
                    >
                        {disclaimer}
                    </Text>
                </div>
            </div>
        </ModalOrBottomSheet>
    );
};

export default SubscriptionModal;
