import React from 'react';

import {IWithClassName} from 'types/withClassName';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Box from 'components/Box/Box';
import Card from 'components/Card/Card';
import Flex from 'components/Flex/Flex';
import Skeleton from 'components/Skeletons/Skeleton/Skeleton';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';
import ButtonSkeleton from 'components/Skeletons/ButtonSkeleton/ButtonSkeleton';

import cx from './SubscriptionSkeleton.scss';

interface ISubscriptionSkeletonProps extends IWithClassName {
    withPromoCode?: boolean;
    type?: 'thin' | 'wide';
    headerSize?: 'xl' | 'xxl';
}

const SubscriptionSkeleton: React.FC<ISubscriptionSkeletonProps> = ({
    withPromoCode,
    type = 'thin',
    headerSize = 'xl',
    className,
}) => {
    const deviceType = useDeviceType();
    const {isDesktop, isMobile} = deviceType;
    const direction = !withPromoCode && isDesktop ? 'row' : 'column';
    const isRow = direction === 'row';

    return (
        <Card
            className={cx(
                'root',
                deviceMods('root', deviceType),
                className,
                type && `root_type_${type}`,
                withPromoCode && `root_withPromoCode`,
            )}
            shadow="default"
        >
            <Flex
                className={cx('wrapper')}
                flexDirection={direction}
                alignItems={isRow ? 'center' : 'stretch'}
                between={isRow ? 10 : 5}
                inline={isRow}
                nowrap={isRow}
            >
                <Box className={cx('column')} between="2">
                    <TextSkeleton className={cx('title')} size={headerSize} />
                    <Box>
                        <TextSkeleton className={cx('subtitle')} size="m" />
                        <TextSkeleton className={cx('subtitle')} size="m" />
                        {isMobile && (
                            <TextSkeleton className={cx('subtitle')} size="m" />
                        )}
                    </Box>
                </Box>

                <Box className={cx('column')} between="2">
                    <Flex
                        inline={isDesktop}
                        between="2"
                        flexDirection={isDesktop ? 'row' : 'column'}
                    >
                        <Skeleton className={cx('input')} />
                        <ButtonSkeleton className={cx('button')} />
                    </Flex>

                    <Box>
                        <TextSkeleton className={cx('description')} size="s" />

                        {isMobile && (
                            <TextSkeleton
                                className={cx('description')}
                                size="s"
                            />
                        )}
                    </Box>
                </Box>
            </Flex>
        </Card>
    );
};

export default SubscriptionSkeleton;
