import React from 'react';

import {ARROW_UP, ARROW_DOWN, ENTER, ESCAPE} from 'constants/eventKeys';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {IBaseOption} from './types/BaseOption';
import {IInputProps} from 'components/Input/types/InputProps';

import Input from 'components/Input/Input';
import {SuggestList} from './components/SuggestList/SuggestList';
import DropdownSuggest from './components/DropdownSuggest/DropdownSuggest';

export interface ISuggestInputProps extends IInputProps {
    onBlur?: (event?: React.FocusEvent<HTMLElement>) => void;
    onFocus?: (event?: React.FocusEvent<HTMLElement>) => void;
}

interface ISuggestProps<IOption extends IBaseOption>
    extends IWithClassName,
        IWithDeviceType {
    containerClassName?: string;

    direction: 'top' | 'bottom';

    defaultValue?: string;
    options: IOption[];
    loading?: boolean;
    inputProps?: ISuggestInputProps;
    preventEnter?: boolean;
    renderInput?: (props: ISuggestInputProps) => React.ReactNode;
    renderOption?: (option: IOption) => React.ReactNode;

    onOptionSelect: (option: IOption) => void;
    onInputChange?: (value: string) => void | Promise<void>;
    onInputFocus?: (value: string) => void | Promise<void>;
    onInputBlur?: (value: string) => void | Promise<void>;
    onShowPopup?: () => void;
}

interface ISuggestState {
    value: string;
    selectedIndex: number;
    contentContainer?: HTMLDivElement;
}

class Suggest<IOption extends IBaseOption> extends React.Component<
    ISuggestProps<IOption>,
    ISuggestState
> {
    readonly state: ISuggestState = {
        value: this.props.defaultValue || '',
        selectedIndex: -1,
    };

    private readonly dropDownRef = React.createRef<DropdownSuggest>();

    static readonly defaultProps = {
        direction: 'bottom',
    };

    componentDidUpdate(prevProps: Readonly<ISuggestProps<IOption>>): void {
        const {defaultValue} = this.props;

        if (defaultValue !== prevProps.defaultValue) {
            this.setState({
                value: defaultValue || '',
            });
        }
    }

    /* Actions */

    private hidePopup(): void {
        this.dropDownRef.current?.hidePopup();
    }

    /* Handlers */

    private handleElementSelect = (index: number): void => {
        const {onOptionSelect, options} = this.props;
        const option = options[index];

        if (option) {
            onOptionSelect(option);
            this.setState({
                value: option.title,
            });
        }

        this.hidePopup();
    };

    private handleItemMouseEnter = (index: number): void => {
        this.setState({
            selectedIndex: index,
        });
    };

    private handleItemMouseLeave = (): void => {
        this.setState({
            selectedIndex: -1,
        });
    };

    private handleActiveElementChange(shift: -1 | 1): void {
        const {selectedIndex} = this.state;
        const {options} = this.props;
        let nextIndex = selectedIndex + shift;

        if (nextIndex < 0) {
            nextIndex = options.length - 1;
        }

        if (nextIndex >= options.length) {
            nextIndex = 0;
        }

        this.setState({
            selectedIndex: nextIndex,
            value: options[nextIndex].title,
        });
    }

    private handleKeyDown = (e: React.KeyboardEvent<HTMLElement>): void => {
        const {preventEnter} = this.props;
        const {selectedIndex} = this.state;
        const {key} = e;

        switch (key) {
            case ARROW_UP: {
                this.handleActiveElementChange(-1);

                break;
            }

            case ARROW_DOWN: {
                this.handleActiveElementChange(1);

                break;
            }

            case ENTER: {
                this.handleElementSelect(selectedIndex);

                if (preventEnter) {
                    e.preventDefault();
                }

                break;
            }

            case ESCAPE: {
                this.hidePopup();

                break;
            }
        }
    };

    private handleInputChange = (value: string): void => {
        const {onInputChange} = this.props;

        this.setState({value});
        onInputChange?.(value);
    };

    private handleHidePopup = (): void => {
        this.setState({
            selectedIndex: -1,
        });
    };

    /* Render */

    private renderSuggestsList = (): React.ReactNode => {
        const {options, renderOption, containerClassName, loading} = this.props;
        const {selectedIndex} = this.state;

        if (!options.length) {
            return null;
        }

        return (
            <SuggestList
                className={containerClassName}
                loading={loading}
                options={options}
                selectedIndex={selectedIndex}
                renderOption={renderOption}
                onItemSelect={this.handleElementSelect}
                onItemMouseEnter={this.handleItemMouseEnter}
                onItemMouseLeave={this.handleItemMouseLeave}
            />
        );
    };

    private renderDefaultInput(props: IInputProps): React.ReactNode {
        return <Input {...props} />;
    }

    render(): React.ReactNode {
        const {
            className,
            deviceType,
            direction,
            inputProps,
            onShowPopup,
            renderInput,
            onInputFocus,
            onInputBlur,
        } = this.props;

        return (
            <DropdownSuggest
                ref={this.dropDownRef}
                className={className}
                deviceType={deviceType}
                value={this.state.value}
                direction={direction}
                inputProps={inputProps}
                onShowPopup={onShowPopup}
                onHidePopup={this.handleHidePopup}
                onInputChange={this.handleInputChange}
                onInputFocus={onInputFocus}
                onInputBlur={onInputBlur}
                onInputKeyDown={this.handleKeyDown}
                renderInput={renderInput || this.renderDefaultInput}
                renderSuggestsList={this.renderSuggestsList}
            />
        );
    }
}

export default Suggest;
