import React from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {IInputChangeEvent} from 'components/Input/types/InputProps';

import {deviceModDesktop} from 'utilities/stylesUtils';
import convertSuggestDirection from 'components/Suggest/components/DropdownSuggest/utilities/convertSuggestDirection';

import Card from 'components/Card/Card';
import {IPopupProps} from 'components/Popup/Popup';
import Dropdown, {IDropdownSwitcherParams} from 'components/Dropdown/Dropdown';

import ScopeContext from 'contexts/ScopeContext';

import {ISuggestInputProps} from '../../Suggest';

import cx from './DropdownSuggest.scss';

export type TDropdownSuggestDirection = 'top' | 'bottom';

interface IDropdownSuggestProps extends IWithClassName, IWithDeviceType {
    direction: TDropdownSuggestDirection;
    value: string;

    inputProps?: ISuggestInputProps;

    renderInput: (props: ISuggestInputProps) => React.ReactNode;
    renderSuggestsList: () => React.ReactNode;

    onShowPopup?: () => void;
    onHidePopup: () => void;
    onInputChange?: (value: string) => void | Promise<void>;
    onInputFocus?: (value: string) => void | Promise<void>;
    onInputBlur?: (value: string) => void;
    onInputKeyDown: (event: React.KeyboardEvent<HTMLElement>) => void;
}

interface IDropdownSuggestState {
    rootContainer?: HTMLDivElement;
}

class DropdownSuggest extends React.Component<
    IDropdownSuggestProps,
    IDropdownSuggestState
> {
    private readonly dropDownRef = React.createRef<Dropdown>();

    readonly state: IDropdownSuggestState = {};

    /* Helpers */

    private setScopeRef = (elem: HTMLDivElement): void => {
        this.setState({rootContainer: elem});
    };

    private getPopupProps = (): Partial<IPopupProps> => {
        const {deviceType, direction} = this.props;

        const commonProps: Partial<IPopupProps> = {
            secondaryOffset: 0,
            plain: true,
            directions: convertSuggestDirection(direction, deviceType),
        };

        if (deviceType.isMobile) {
            return {
                mainOffset: 8,
                ...commonProps,
            };
        }

        return {
            mainOffset: 4,
            ...commonProps,
        };
    };

    /* Actions */

    private showPopup(): void {
        if (this.dropDownRef && this.dropDownRef.current) {
            this.dropDownRef.current.showPopup();
        }
    }

    hidePopup(): void {
        if (this.dropDownRef && this.dropDownRef.current) {
            this.dropDownRef.current.hidePopup();
        }
    }

    /* Handlers */

    private handleInputChange = (_: IInputChangeEvent, value: string): void => {
        const {onInputChange} = this.props;

        onInputChange?.(value);
        this.showPopup();
    };

    private handleInputFocus = (): void => {
        const {onInputFocus, value} = this.props;

        this.showPopup();

        if (onInputFocus) {
            onInputFocus(value);
        }
    };

    private handleInputBlur = (): void => {
        const {onInputBlur, value} = this.props;

        if (onInputBlur) {
            onInputBlur(value);
        }
    };

    /* Render */

    private renderSuggests = ({
        meta,
    }: {
        meta: {visible: boolean};
    }): React.ReactNode => {
        if (!meta.visible) {
            return null;
        }

        return (
            <Card className={cx('card')} shadow="popup">
                {this.props.renderSuggestsList()}
            </Card>
        );
    };

    private renderPopupInput = ({
        input,
    }: IDropdownSwitcherParams<unknown>): React.ReactNode => {
        const {value, inputProps, renderInput, onInputKeyDown} = this.props;

        const inputRenderProps = {
            ...inputProps,
            inputRef: input.ref,
            value: value,
            onChange: this.handleInputChange,
            onFocus: this.handleInputFocus,
            onBlur: this.handleInputBlur,
            onKeyDown: onInputKeyDown,
        };

        return renderInput(inputRenderProps);
    };

    render(): React.ReactNode {
        const {className, deviceType, onShowPopup, onHidePopup} = this.props;

        return (
            <div className={cx('popupScope', className)} ref={this.setScopeRef}>
                <ScopeContext.Provider value={this.state.rootContainer}>
                    <Dropdown
                        popupClassName={cx(
                            'popup',
                            deviceModDesktop('popup', deviceType),
                        )}
                        ref={this.dropDownRef}
                        isModalView={false}
                        popupProps={this.getPopupProps()}
                        renderPopup={this.renderSuggests}
                        renderSwitcher={this.renderPopupInput}
                        onShowPopup={onShowPopup}
                        onHidePopup={onHidePopup}
                    />
                </ScopeContext.Provider>
            </div>
        );
    }
}

export default DropdownSuggest;
