import React, {PureComponent, ReactNode, createRef} from 'react';

import {IWithClassName} from 'types/withClassName';

import Flex from 'components/Flex/Flex';

import cx from './Swipeable.scss';

const DURATION_INSTANT = 200; // scss constant $DURATION_INSTANT: 0.2s;
const MOVE_START_THRESHOLD = 20;

export interface ISwipeableRenderItemParams<T> {
    data: T;
    index: number;
    isCurrent: boolean;
}
interface ISwipeableProps<T> extends IWithClassName {
    renderItem: (params: ISwipeableRenderItemParams<T>) => ReactNode;
    itemsData: T[];
    onSideSwipe?: (index: number) => void;
    onDownSwipe?: () => void;
    index?: number;
    sideThreshold: number;
    downThreshold: number;
    style?: React.CSSProperties;
    contentClassName?: string;
    showNav?: boolean;
    renderNav?: (items: any[], index: number) => ReactNode;
    disableSwipe?: boolean;
    itemVerticalAlignment?: React.CSSProperties['alignItems'];
    spacing?: 'l' | 'xs';
    withSwipeDown?: boolean;
}

interface ISwipeableState {
    currentIndex: number;
    currentItemNode: ReactNode;
    prevItemNode: ReactNode;
    nextItemNode: ReactNode;
    isMoving: boolean;
    hasNext: boolean;
    hasPrev: boolean;
    deltaX: number;
    deltaY: number;
    swipeFinishingTranslate?: string;
    itemKeys: string[];
}

type TSwipeFinished = 'left' | 'right' | 'down' | undefined;

export default class Swipeable<T> extends PureComponent<
    ISwipeableProps<T>,
    ISwipeableState
> {
    readonly state: ISwipeableState = {
        currentIndex: 0,
        currentItemNode: this.props.itemsData?.length
            ? this.renderItem(0, true)
            : null,
        prevItemNode: null,
        nextItemNode:
            this.props.itemsData?.length > 1 ? this.renderItem(1, false) : null,
        isMoving: false,
        hasNext: false,
        hasPrev: false,
        deltaX: 0,
        deltaY: 0,
        swipeFinishingTranslate: undefined,
        itemKeys: this.props.itemsData?.length
            ? this.props.itemsData.map(this.getItemKey)
            : [],
    };

    private timeout: NodeJS.Timer | undefined;

    private readonly rootRef = createRef<HTMLDivElement>();
    private readonly contentRef = createRef<HTMLDivElement>();

    private swipeStartPageX?: number;
    private swipeStartPageY?: number;

    private isHorizontalMoving: boolean = false;
    private isVerticalMoving: boolean = false;

    static defaultProps = {
        sideThreshold: 0.5,
        downThreshold: 0.1,
    };

    componentDidMount(): void {
        const {current: root} = this.rootRef;

        if (root && window) {
            root.addEventListener('touchstart', this.handlePanStart);
            window.addEventListener('touchmove', this.handlePanHorizontal);
            window.addEventListener('touchend', this.handlePanEnd);
        }

        const {currentIndex} = this.state;
        const {index = currentIndex} = this.props;

        this.updateCurrentItem(index);
        this.updateItemKeys(this.props.itemsData);
    }

    componentDidUpdate(prevProps: ISwipeableProps<T>): void {
        const index = Number(this.props.index);

        if (
            index >= 0 &&
            index !== Number(prevProps.index) &&
            index !== this.state.currentIndex
        ) {
            this.updateCurrentItem(index);
        }

        if (this.props.itemsData !== prevProps.itemsData) {
            this.updateItemKeys(this.props.itemsData);
        }
    }

    componentWillUnmount(): void {
        const {current: root} = this.rootRef;

        if (root && window) {
            root.removeEventListener('touchstart', this.handlePanStart);
            window.removeEventListener('touchmove', this.handlePanHorizontal);
            window.removeEventListener('touchend', this.handlePanEnd);
        }

        if (this.timeout) {
            clearTimeout(this.timeout);
        }
    }

    private handlePanStart = (e: TouchEvent): void => {
        if (this.props.disableSwipe || e.touches.length !== 1) {
            return;
        }

        this.swipeStartPageX = e.touches[0].pageX;
        this.swipeStartPageY = e.touches[0].pageY;
        this.isHorizontalMoving = false;
        this.isVerticalMoving = false;
    };

    private handlePanHorizontal = (e: TouchEvent): void => {
        if (this.props.disableSwipe) {
            return;
        }

        if (
            this.swipeStartPageX === undefined &&
            this.swipeStartPageY === undefined
        ) {
            return;
        }

        if (e.touches.length !== 1) {
            this.swipeStartPageX = undefined;
            this.swipeStartPageY = undefined;

            return;
        }

        if (this.isSwipeFinishing()) {
            return;
        }

        const {deltaX, deltaY} = this.getDelta(e.touches[0]);

        this.setState({
            deltaX,
            deltaY,
            isMoving: true,
        });
    };

    private handlePanEnd = (): void => {
        if (this.swipeStartPageX === undefined) {
            return;
        }

        const {itemsData} = this.props;

        if (this.isSwipeFinishing()) {
            return;
        }

        const finished = this.tryGetSwipeFinished();

        if (!finished) {
            this.stopMoving();

            return;
        }

        const nextIndex = this.getNextIndex(finished);

        if (!itemsData[nextIndex]) {
            this.stopMoving();

            return;
        }

        this.finishSwipe(finished, nextIndex);
    };

    private getNextIndex(finished: TSwipeFinished): number {
        const {currentIndex} = this.state;

        switch (finished) {
            case 'left':
                return currentIndex + 1;
            case 'right':
                return currentIndex - 1;
            default:
                return currentIndex;
        }
    }

    private updateItemKeys(itemsData: T[]): void {
        this.setState({itemKeys: itemsData.map(this.getItemKey)});
    }

    private getItemKey(itemData: T, index: number): string {
        if (typeof itemData === 'string') {
            return itemData;
        }

        if (typeof itemData === 'number' || typeof itemData === 'boolean') {
            return itemData.toString();
        }

        if (typeof itemData === 'object') {
            return Object.values(itemData).join('');
        }

        return index.toString();
    }

    private updateCurrentItem(
        newCurrentIndex: number,
        setStateCallback?: () => void,
    ): void {
        const {itemsData} = this.props;
        const currentItemNode = this.getNewCurrentItem(newCurrentIndex);
        const hasNext = itemsData.length - 1 > newCurrentIndex;
        const hasPrev = newCurrentIndex > 0;
        const nextItemNode = this.getNewNextItem(newCurrentIndex, hasNext);
        const prevItemNode = this.getNewPrevItem(newCurrentIndex, hasPrev);

        this.setState(
            {
                currentIndex: newCurrentIndex,
                currentItemNode,
                hasNext,
                hasPrev,
                nextItemNode,
                prevItemNode,
            },
            setStateCallback,
        );
    }

    private getNewCurrentItem(newCurrentIndex: number): ReactNode {
        return this.renderItem(newCurrentIndex, true);
    }

    private getNewNextItem(
        newCurrentIndex: number,
        hasNext: boolean,
    ): ReactNode {
        if (!hasNext) {
            return null;
        }

        return this.renderItem(newCurrentIndex + 1, false);
    }

    private getNewPrevItem(
        newCurrentIndex: number,
        hasPrev: boolean,
    ): ReactNode {
        if (!hasPrev) {
            return null;
        }

        return this.renderItem(newCurrentIndex - 1, false);
    }

    private renderItem(index: number, isCurrent: boolean): ReactNode {
        const {renderItem, itemsData} = this.props;
        const currentItem = itemsData[index];

        return renderItem({data: currentItem, index, isCurrent});
    }

    private getDelta({pageX, pageY}: {pageX: number; pageY: number}): {
        deltaX: number;
        deltaY: number;
    } {
        const deltaX = this.getDeltaX(pageX);
        const deltaY = this.getDeltaY(pageY);

        if (!this.isVerticalMoving && MOVE_START_THRESHOLD < Math.abs(deltaX)) {
            this.isHorizontalMoving = true;
        } else if (
            !this.isHorizontalMoving &&
            MOVE_START_THRESHOLD < Math.abs(deltaY)
        ) {
            this.isVerticalMoving = true;
        }

        if (this.isHorizontalMoving) {
            return {
                deltaX,
                deltaY: 0,
            };
        } else if (this.isVerticalMoving) {
            return {
                deltaX: 0,
                deltaY,
            };
        }

        return {
            deltaX: 0,
            deltaY: 0,
        };
    }

    private getDeltaX(pageX: number): number {
        const {current: content} = this.contentRef;
        const {sideThreshold} = this.props;

        if (!content || this.swipeStartPageX === undefined) {
            return 0;
        }

        const deltaX = pageX - this.swipeStartPageX;

        if (deltaX < 0) {
            if (this.state.hasNext) {
                return Math.max(deltaX, -content.offsetWidth);
            }

            return Math.max(deltaX, -content.offsetWidth * sideThreshold);
        }

        if (this.state.hasPrev) {
            return Math.min(deltaX, content.offsetWidth);
        }

        return Math.min(deltaX, content.offsetWidth * sideThreshold);
    }

    private getDeltaY(pageY: number): number {
        const {withSwipeDown} = this.props;
        const {current: content} = this.contentRef;

        if (!withSwipeDown) {
            return 0;
        }

        if (!content || this.swipeStartPageY === undefined) {
            return 0;
        }

        const deltaY = pageY - this.swipeStartPageY;

        if (deltaY < 0) {
            return 0;
        }

        return Math.min(deltaY, content.offsetHeight);
    }

    private tryGetSwipeFinished(): TSwipeFinished {
        const {current: content} = this.contentRef;

        if (!content) {
            return undefined;
        }

        const {deltaX, deltaY} = this.state;
        const {sideThreshold, downThreshold} = this.props;

        if (deltaX > content.offsetWidth * sideThreshold) {
            return 'right';
        }

        if (deltaX < -content.offsetWidth * sideThreshold) {
            return 'left';
        }

        if (deltaY > content.offsetHeight * downThreshold) {
            return 'down';
        }

        return undefined;
    }

    private finishSwipe(finished: TSwipeFinished, nextIndex: number): void {
        const {onSideSwipe, onDownSwipe} = this.props;
        const swipeFinishingTranslate =
            this.getSwipeFinishingTranslate(finished);

        this.setState({
            swipeFinishingTranslate,
            deltaX: 0,
            deltaY: 0,
        });
        this.swipeStartPageX = undefined;
        this.swipeStartPageY = undefined;
        this.timeout = setTimeout(() => {
            this.updateCurrentItem(nextIndex, () =>
                this.setState({
                    isMoving: false,
                    swipeFinishingTranslate: undefined,
                }),
            );

            if (finished === 'left' || finished === 'right') {
                onSideSwipe?.(nextIndex);
            } else if (finished === 'down') {
                onDownSwipe?.();
            }
        }, DURATION_INSTANT);
    }

    private stopMoving(): void {
        this.swipeStartPageX = undefined;
        this.swipeStartPageY = undefined;
        this.setState({
            deltaX: 0,
            deltaY: 0,
        });
        this.timeout = setTimeout(() => {
            this.setState({
                isMoving: false,
            });
        }, DURATION_INSTANT);
    }

    private getSwipeFinishingTranslate(finished: TSwipeFinished): string {
        if (finished === 'right') {
            return 'translateX(100%)';
        }

        if (finished === 'left') {
            return 'translateX(-100%)';
        }

        if (finished === 'down') {
            const {deltaY} = this.state;

            return `translateY(${deltaY}px)`;
        }

        return '';
    }

    private isSwipeFinishing(): boolean {
        return Boolean(this.state.swipeFinishingTranslate);
    }

    private getContentStyle(): object {
        const {deltaX, deltaY, swipeFinishingTranslate} = this.state;

        if (swipeFinishingTranslate) {
            return {
                transform: swipeFinishingTranslate,
            };
        }

        return {
            transform: `translateX(${deltaX}px) translateY(${deltaY}px)`,
        };
    }

    private renderNav(items: any[], currentIndex: number): ReactNode {
        return (
            <div className={cx('nav')}>
                {items.map((i, index: number) => {
                    return (
                        <div
                            key={index}
                            className={cx('nav-item', {
                                'nav-item-active': index === currentIndex,
                            })}
                        />
                    );
                })}
            </div>
        );
    }

    render(): ReactNode {
        const {rootRef, contentRef} = this;
        const {
            contentClassName,
            style,
            className,
            showNav,
            itemVerticalAlignment = 'center',
            spacing = 'l',
            renderNav,
        } = this.props;
        const {
            isMoving,
            currentIndex,
            currentItemNode,
            prevItemNode,
            nextItemNode,
            itemKeys,
        } = this.state;

        return (
            <div className={cx('root', className)} style={style} ref={rootRef}>
                <div
                    className={cx('content', contentClassName, {
                        content_moving: isMoving,
                        content_finished: this.isSwipeFinishing(),
                    })}
                    ref={contentRef}
                    style={this.getContentStyle()}
                >
                    <Flex
                        key={itemKeys[currentIndex - 1]}
                        className={cx(
                            'item',
                            'prev',
                            `prev_spacing_${spacing}`,
                        )}
                        alignItems={itemVerticalAlignment}
                        justifyContent="center"
                    >
                        {prevItemNode}
                    </Flex>

                    <Flex
                        key={itemKeys[currentIndex]}
                        className={cx('item')}
                        alignItems={itemVerticalAlignment}
                        justifyContent="center"
                    >
                        {currentItemNode}
                    </Flex>

                    <Flex
                        key={itemKeys[currentIndex + 1]}
                        className={cx(
                            'item',
                            'next',
                            `next_spacing_${spacing}`,
                        )}
                        alignItems={itemVerticalAlignment}
                        justifyContent="center"
                    >
                        {nextItemNode}
                    </Flex>
                </div>
                {showNav && renderNav
                    ? renderNav(itemKeys, currentIndex)
                    : this.renderNav(itemKeys, currentIndex)}
            </div>
        );
    }
}
