import React, {useCallback, useEffect, useMemo, useRef, useState} from 'react';
import stubFalse from 'lodash/stubFalse';

import {IWithClassName} from 'types/withClassName';
import {IRenderTabProps} from 'components/Tabs/types/IRenderTabProps';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import {TBoxSizes} from 'components/Box/Box';
import useGetPropsForRenderTab from 'components/Tabs/hooks/useGetPropsForRenderTab';
import useScrollToActiveTab from 'components/Tabs/hooks/useScrollToActiveTab';
import Intersperse from 'components/Intersperse/Intersperse';
import Flex from 'components/Flex/Flex';
import useBetween from 'components/Tabs/hooks/useBetween';
import Tab from './components/Tab/Tab';
import useRenderSeparator from 'components/Tabs/hooks/useRenderSeparator';

import cx from './Tabs.scss';

export interface ITabsProps<TabType, TabIdType>
    extends IWithClassName,
        IWithQaAttributes {
    activeTabId?: TabIdType;
    tabs: TabType[];

    scrollableClassName?: string;
    padderClassName?: string;
    flexClassName?: string;

    offset?: TBoxSizes;
    between?: TBoxSizes;
    arrow?: boolean;
    bold?: boolean;
    border?: boolean;

    additional?: React.ReactNode;

    /**
     * Metadata props
     */
    itemScope?: boolean;
    itemType?: string;

    renderTab(
        renderTabProps: IRenderTabProps<TabType, TabIdType>,
    ): React.ReactNode;
    getTabId(tab: TabType): TabIdType;
    getTabIsDisabled?(tab: TabType): boolean;
    getTabIsActive?(tab: TabType): boolean;
    onChange?(tab: TabType, tabId: TabIdType): void;
}

function Tabs<TabType, TabIdType>(
    props: ITabsProps<TabType, TabIdType>,
): React.ReactElement {
    const {
        tabs,
        className,
        padderClassName,
        scrollableClassName,
        flexClassName,
        offset,
        renderTab,
        arrow = false,
        between,
        border = false,
        additional = null,
        bold = false,
        getTabId,
        getTabIsDisabled = stubFalse,
        getTabIsActive,
        itemScope,
        itemType,
        onChange,
        activeTabId: activeTabIdFromProps,
    } = props;

    const deviceType = useDeviceType();

    const activeTabRef = useRef<HTMLButtonElement>(null);
    const scrollContainerRef = useRef<HTMLDivElement>(null);

    const [activeTabId, setActiveTabId] = useState(
        activeTabIdFromProps || getTabId(tabs[0]),
    );

    useEffect(() => {
        if (activeTabIdFromProps) {
            setActiveTabId(activeTabIdFromProps);
        }
    }, [activeTabIdFromProps]);

    const checkActiveTab = useCallback(
        (tab: TabType, tabId: TabIdType): boolean => {
            if (getTabIsActive) {
                return getTabIsActive(tab);
            }

            return tabId === activeTabId;
        },
        [activeTabId, getTabIsActive],
    );

    const handleTabClick = useCallback(
        (tab: TabType, tabId: TabIdType): void => {
            const isActive = checkActiveTab(tab, tabId);

            setActiveTabId(tabId);

            if (!isActive && onChange) {
                onChange(tab, tabId);
            }
        },
        [checkActiveTab, onChange],
    );

    useScrollToActiveTab({activeTabRef, activeTabId, scrollContainerRef});

    const defaultBetween = useBetween({deviceType, arrow, bold});

    const getPropsForRenderTab = useGetPropsForRenderTab({
        activeTabRef,
        checkActiveTab,
        handleTabClick,
        bold,
    });
    const content = useMemo(() => {
        return tabs.map((tab: TabType): React.ReactNode => {
            const tabId = getTabId(tab);
            const tabDisabled = getTabIsDisabled(tab);

            return renderTab(getPropsForRenderTab(tab, tabId, tabDisabled));
        });
    }, [getPropsForRenderTab, getTabId, getTabIsDisabled, renderTab, tabs]);

    const renderSeparator = useRenderSeparator({tabs, getTabIsDisabled});

    return (
        <HorizontalScroller
            className={cx(className, 'root', {root_border: border})}
            scrollerRef={scrollContainerRef}
            scrollableClassName={scrollableClassName}
            padderClassName={cx(padderClassName, 'padder')}
            offset={offset}
            itemScope={itemScope}
            itemType={itemType}
            {...prepareQaAttributes(props)}
        >
            <Flex
                className={cx(flexClassName, 'tabs')}
                inline
                between={between ?? defaultBetween}
                alignItems="baseline"
                nowrap
            >
                {arrow ? (
                    <Intersperse separator={renderSeparator}>
                        {content}
                    </Intersperse>
                ) : (
                    content
                )}
                {additional && (
                    <div className={cx('additional')}>{additional}</div>
                )}
            </Flex>
        </HorizontalScroller>
    );
}

Tabs.Tab = Tab;

export default Tabs;
