import React, {useEffect, useRef} from 'react';

import {ANIMATION_DURATION_S} from 'constants/animations';

import {animate} from 'utilities/animation/animate';
import {makeEaseInOut} from 'utilities/animation/animationMods';
import {animationLinear} from 'utilities/animation/animationTypes';

interface IScrollToActiveTab<TabType> {
    activeTabRef: React.RefObject<HTMLElement>;
    scrollContainerRef: React.RefObject<HTMLDivElement>;
    activeTabId: TabType;
}

const scrollDistanceBound = (
    position: number,
    min: number,
    max: number,
): number => Math.min(Math.max(position, min), max);

export default function useScrollToActiveTab<TabType>({
    activeTabRef,
    scrollContainerRef,
    activeTabId,
}: IScrollToActiveTab<TabType>): void {
    const animationIdRef = useRef<number | null>(null);

    useEffect(() => {
        const scrollContainerNode = scrollContainerRef.current;
        const activeTabNode = activeTabRef.current;

        if (
            !(activeTabNode instanceof Element) ||
            !(scrollContainerNode instanceof Element)
        ) {
            return;
        }

        const activeTabLeft = activeTabNode.offsetLeft;
        const activeTabWidth = activeTabNode.offsetWidth;
        const containerWidth = scrollContainerNode.offsetWidth;
        const containerScrollWidth = scrollContainerNode.scrollWidth;
        const containerScrollLeft = scrollContainerNode.scrollLeft;
        const maxScrollDistance = containerScrollWidth - containerWidth;

        if (maxScrollDistance <= 0) {
            return;
        }

        const nextScrollLeft = scrollDistanceBound(
            activeTabLeft - (containerWidth - activeTabWidth) / 2,
            0,
            maxScrollDistance,
        );
        const scrollDiff = nextScrollLeft - containerScrollLeft;
        const animationId = animationIdRef.current;

        if (animationId) {
            cancelAnimationFrame(animationId);

            animationIdRef.current = null;
        }

        animationIdRef.current = animate({
            duration: ANIMATION_DURATION_S,
            timing: makeEaseInOut(animationLinear),
            draw(progress: number) {
                scrollContainerNode.scrollLeft =
                    containerScrollLeft + scrollDiff * progress;
            },
        });
    }, [activeTabId, activeTabRef, scrollContainerRef]);
}
