/* eslint-disable no-use-before-define */
import React from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import cx from './Text.scss';

export interface ITextProps extends IWithQaAttributes, IWithClassName {
    /**
     * Вес текста
     *
     * @default 'normal'
     */
    weight?: TTextWeight;
    /**
     * Размер текста
     *
     * @default 'm'
     */
    size?: TTextSize;
    /**
     * Цвет текста
     *
     * @default 'primary'
     */
    color?: TextColor;
    disabled?: boolean;
    whiteSpace?: WhiteSpace;
    align?: TTextAlign;
    onClick?: () => void;
    tag?: keyof JSX.IntrinsicElements;
    tagProps?: React.HTMLAttributes<any> & React.PropsWithRef<any>;
    overflow?: TOverflow;
}

const Text: React.FC<ITextProps> = ({
    className,
    weight = 'normal',
    size = 'm',
    color = 'primary',
    disabled,
    whiteSpace,
    align,
    onClick,
    children,
    tag,
    tagProps,
    overflow,
    ...rest
}) => {
    const Tag = tag || 'span';

    return (
        <Tag
            {...prepareQaAttributes(rest)}
            onClick={onClick}
            {...tagProps}
            className={cx(
                className,
                textSizeCn(size),
                textWeightCn(weight),
                textColorCn(color),
                whiteSpace ? whiteSpaceCn(whiteSpace) : undefined,
                alignCn(align),
                overflowCn(overflow),
                disabled && 'disabled',
            )}
        >
            {children}
        </Tag>
    );
};

export default Text;

export type TextColor =
    | 'current'
    | 'primary'
    | 'secondary'
    | 'inverse'
    | 'link'
    | 'link-hover'
    | 'success'
    | 'highlight'
    | 'alert'
    | 'link-pale'
    | 'plus'
    | 'plus-cropped';

export type TTextSize =
    | 'xs'
    | 's'
    | 's-inset'
    | 'm'
    | 'l'
    | 'xl'
    | 'xxl'
    | 'inherit';

export type TTextWeight = 'normal' | 'medium' | 'bold';

export type WhiteSpace = 'normal' | 'nowrap' | 'pre-wrap';

export type TTextAlign = 'left' | 'right' | 'center' | 'justify';

export type TOverflow = 'ellipsis';

export const textSizeCn = (size: TTextSize) => cx(`size_${size}`);

export const textWeightCn = (weight: TTextWeight) => cx(`weight_${weight}`);

export const textColorCn = (color: TextColor) => cx(`color_${color}`);

export const whiteSpaceCn = (whitePace: WhiteSpace) =>
    cx(`whiteSpace_${whitePace}`);

const alignCn = (align: TTextAlign | undefined): string | undefined =>
    align ? cx(`align_${align}`) : undefined;

const overflowCn = (overflow: TOverflow | undefined): string | undefined =>
    overflow ? cx(`overflow_${overflow}`) : undefined;
