import {
    ReactElement,
    ComponentType,
    ReactNode,
    RefObject,
    MouseEventHandler,
} from 'react';

import {IIconProps} from 'icons/types/icon';
import {IWithClassName} from 'src/types/withClassName';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import cx from './TextWithIcon.scss';

export type TSize = 'xs' | 's' | 's-inset' | 'm' | 'l' | 'xl';

export type TIconSize = 12 | 16 | 24 | 36;

type TLeftIconGenericProps<T extends IIconProps> = IIconProps extends T
    ? {
          iconLeftProps?: T;
      }
    : {
          iconLeftProps: T;
      };

type TRightIconGenericProps<T extends IIconProps> = IIconProps extends T
    ? {
          iconRightProps?: T;
      }
    : {
          iconRightProps: T;
      };

export type TTextWithIconProps<
    TLeftIconProps extends IIconProps,
    TRightIconProps extends IIconProps,
> = IWithClassName &
    IWithQaAttributes &
    TLeftIconGenericProps<TLeftIconProps> &
    TRightIconGenericProps<TRightIconProps> & {
        iconLeft?: ComponentType<TLeftIconProps>;
        iconLeftRef?: RefObject<HTMLSpanElement>;
        iconLeftClassName?: string;
        iconRight?: ComponentType<TRightIconProps>;
        iconRightRef?: RefObject<HTMLSpanElement>;
        iconRightClassName?: string;
        text: ReactNode;
        textClassName?: string;
        size?: TSize;
        spaceBetween?: '1' | '2' | '3' | 1 | 2 | 3;
        onClick?: MouseEventHandler<HTMLElement>;
        iconSize?: TIconSize;
    };

function TextWithIcon<
    TLeftIconProps extends IIconProps = IIconProps,
    TRightIconProps extends IIconProps = IIconProps,
>(props: TTextWithIconProps<TLeftIconProps, TRightIconProps>): ReactElement {
    const {
        className = '',
        iconLeft: IconLeft,
        iconLeftRef,
        iconLeftProps,
        iconLeftClassName,
        iconRight: IconRight,
        iconRightRef,
        iconRightProps,
        iconRightClassName,
        size = 'm',
        spaceBetween = 2,
        text,
        textClassName,
        onClick,
        iconSize,
    } = props;

    return (
        <div
            className={cx(
                'content',
                `content_size_${size}`,
                iconSize && `iconSize_${iconSize}`,
                className,
            )}
            onClick={onClick}
            {...prepareQaAttributes({
                parent: props,
            })}
        >
            {IconLeft && (
                <span
                    className={cx(
                        'icon',
                        'icon_left',
                        `space_${spaceBetween}`,
                        iconLeftClassName,
                    )}
                    ref={iconLeftRef}
                >
                    {/* @ts-ignore - typescript раздельно смешивает тип для IconLeft и iconLeftProps. TS пока не умеет понимать, что при generic иконке, iconLeftProps не будет undefined. Но это обеспечивается внешним API */}
                    <IconLeft {...iconLeftProps} />
                </span>
            )}
            <span className={cx('text', textClassName)}>{text}</span>
            {IconRight && (
                <span
                    className={cx(
                        'icon',
                        'icon_right',
                        `space_${spaceBetween}`,
                        iconRightClassName,
                    )}
                    ref={iconRightRef}
                >
                    {/* @ts-ignore - typescript раздельно смешивает тип для IconRight и iconRightProps. TS пока не умеет понимать, что при generic иконке, iconRightProps не будет undefined. Но это обеспечивается типизацией внешнего API */}
                    <IconRight {...iconRightProps} />
                </span>
            )}
        </div>
    );
}

export default TextWithIcon;
