import React from 'react';
import _noop from 'lodash/noop';
import {throttle} from 'lodash';

import EPopupDirection from 'components/Popup/types/EPopupDirection';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import {IPopupProps} from 'components/Popup/Popup';
import Dropdown from 'components/Dropdown/Dropdown';
import TravellersCount from 'components/TravellersCount/TravellersCount';
import RotatingArrowIcon from 'components/RotatingArrowIcon/RotatingArrowIcon';
import SearchFormFieldModal from 'components/SearchFormFieldModal/SearchFormFieldModal';
import SelectArrow from 'icons/16/SelectArrow';
import TravellersSelect from './components/TravellersSelect/TravellersSelect';

import cx from './Travellers.scss';

type TTriggerViewType = 'union' | 'tile';

type TSetRef = (node: HTMLElement | null) => void;

interface ITravellersProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    type: string;
    triggerClassName?: string;
    triggerFocusClassName?: string;
    children: React.ReactNode;
    tabIndex?: number;
    isModalView: boolean;
    triggerViewType: TTriggerViewType;
    size: 's' | 'm' | 'xl';
    canToggle?: boolean;
    triggerRef(): void;
    buttonComponent(): React.ReactNode;
    onShowPopup(): void;
    onHidePopup(): void;
    onFocus(): void;
    onBlur(): void;
}

interface ITravellersState {
    triggerWidth?: number;
}

const POPUP_BORDER_WIDTH = 2;

class Travellers extends React.Component<ITravellersProps, ITravellersState> {
    private triggerNode: HTMLElement | null = null;

    private resetTriggerWidth: () => void = throttle(() => {
        const {deviceType} = this.props;

        if (this.triggerNode && deviceType.isDesktop) {
            this.setState({
                triggerWidth: this.triggerNode.clientWidth - POPUP_BORDER_WIDTH,
            });
        }
    }, 100);

    static readonly defaultProps = {
        type: '',
        isModalView: false,
        triggerViewType: 'union',
        size: 'xl',
        triggerRef: _noop,
        onShowPopup: _noop,
        onHidePopup: _noop,
        onFocus: _noop,
        onBlur: _noop,
    };

    readonly state: ITravellersState = {};

    componentDidMount(): void {
        window.addEventListener('resize', this.onWindowResize);
        this.resetTriggerWidth();
    }

    componentWillUnmount(): void {
        window.removeEventListener('resize', this.onWindowResize);
    }

    static TravellersCount = TravellersCount;
    static TravellersSelect = TravellersSelect;

    private onWindowResize = (): void => {
        this.resetTriggerWidth();
    };

    private setRootNodeRef =
        (refs: TSetRef[]) =>
        (rootNode: HTMLElement | null): void => {
            this.triggerNode = rootNode;
            refs.forEach(ref => ref(rootNode));
        };

    private getPopupProps = (): Partial<IPopupProps> => {
        const {isModalView, size} = this.props;

        if (isModalView) {
            return {};
        }

        return {
            directions: [
                EPopupDirection.BOTTOM_RIGHT,
                EPopupDirection.BOTTOM_LEFT,
            ],
            plain: true,
            nonvisual: false,
            mainOffset: size === 'xl' ? undefined : 4,
        };
    };

    private renderTravellersTrigger = ({
        refs,
        input,
        visible,
        triggerViewType,
        isModalButton,
        isMobile,
    }: {
        refs: TSetRef[];
        input: {
            onKeyDown?: React.KeyboardEventHandler<HTMLButtonElement>;
            onFocus?: React.FocusEventHandler<HTMLButtonElement>;
            onClick?: React.MouseEventHandler<HTMLButtonElement>;
            onMouseDown?: React.MouseEventHandler<HTMLButtonElement>;
        };
        visible?: boolean;
        triggerViewType?: TTriggerViewType;
        isModalButton?: boolean;
        isMobile?: boolean;
    }): React.ReactElement => {
        const {
            tabIndex,
            buttonComponent,
            triggerClassName,
            triggerFocusClassName,
            size,
        } = this.props;
        const {onKeyDown, onFocus, onClick, onMouseDown} = input;

        return (
            <button
                className={cx(
                    triggerClassName,
                    'button',
                    `button_${size}`,
                    {
                        button_focus: visible,
                        button_modal: isModalButton,
                        button_mobile: isMobile,
                        [`button_viewType_${triggerViewType}`]: triggerViewType,
                    },
                    visible && triggerFocusClassName,
                )}
                type="button"
                tabIndex={tabIndex}
                onFocus={onFocus}
                onKeyDown={onKeyDown}
                onMouseDown={onMouseDown}
                // Только для тачей, см https://st.yandex-team.ru/TRAVELFRONT-4622
                onClick={isMobile ? onClick : undefined}
                ref={this.setRootNodeRef(refs)}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'trigger',
                })}
            >
                <div className={cx('buttonContent')}>{buttonComponent()}</div>
                {this.renderIcon({isModalButton, visible})}
            </button>
        );
    };

    private renderIcon({
        visible,
        isModalButton,
    }: {
        visible: boolean | undefined;
        isModalButton: boolean | undefined;
    }): React.ReactNode {
        const {size} = this.props;

        if (isModalButton) {
            return null;
        }

        if (size === 'xl') {
            return (
                <div className={cx('buttonIcon')}>
                    <RotatingArrowIcon
                        className={cx('icon')}
                        rotated={Boolean(visible)}
                    />
                </div>
            );
        }

        return <SelectArrow />;
    }

    /* Modal View */

    private renderModalTravellers = ({
        meta,
    }: {
        meta: {hidePopup: () => void};
    }): React.ReactElement => {
        const {type} = this.props;
        const {hidePopup} = meta;
        const componentNode = this.renderTravellers();
        const triggerNode = this.renderTravellersTrigger({
            input: {},
            refs: [],
            isModalButton: true,
        });

        return (
            <SearchFormFieldModal
                triggerNode={triggerNode}
                componentNode={componentNode}
                fieldType={type}
                cancelButtonIsVisible={false}
                isLastControl={type === 'travellers'}
                onHideModal={hidePopup}
                onCancelClick={_noop}
            />
        );
    };

    /* Dropdown View */

    private renderTravellers = (): React.ReactElement => {
        const {isModalView} = this.props;
        const {triggerWidth} = this.state;

        return (
            <div
                className={cx('travellers', {
                    travellers_modal: isModalView,
                })}
                style={{width: triggerWidth}}
            >
                {this.props.children}
            </div>
        );
    };

    private renderDropdownTrigger = ({
        meta,
        input,
    }: {
        input: {
            ref: TSetRef;
            onKeyDown?: React.KeyboardEventHandler<HTMLButtonElement>;
            onFocus?: React.FocusEventHandler<HTMLButtonElement>;
            onClick?: React.MouseEventHandler<HTMLButtonElement>;
            onMouseDown?: React.MouseEventHandler<HTMLButtonElement>;
        };
        meta: {visible: boolean};
    }): React.ReactElement => {
        const {
            triggerRef,
            triggerViewType,
            deviceType: {isMobile},
        } = this.props;
        const {visible} = meta;
        const {ref} = input;

        return this.renderTravellersTrigger({
            input,
            visible,
            isMobile,
            triggerViewType,
            refs: [ref, triggerRef],
        });
    };

    render(): React.ReactNode {
        const {
            isModalView,
            onFocus,
            onBlur,
            onShowPopup,
            onHidePopup,
            canToggle,
        } = this.props;
        const popupProps = this.getPopupProps();

        return (
            <Dropdown
                popupProps={popupProps}
                onShowPopup={onShowPopup}
                onHidePopup={onHidePopup}
                popupClassName={cx('popupContainer')}
                popupComponent={
                    isModalView
                        ? this.renderModalTravellers
                        : this.renderTravellers
                }
                switcherComponent={this.renderDropdownTrigger}
                isModalView={isModalView}
                onFocus={onFocus}
                onBlur={onBlur}
                canToggle={canToggle}
                popupMotionless={true}
            />
        );
    }
}

export default Travellers;
