import React, {ReactNode, useCallback, useMemo, useRef} from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import useTooltipHover from 'utilities/hooks/useTooltipHover';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import PlusIcon from 'icons/16/Plus';
import MinusIcon from 'icons/16/Minus';
import Button from 'components/Button/Button';
import MessageBoxPopupOrBottomSheet from 'components/MessageBoxPopupOrBottomSheet/MessageBoxPopupOrBottomSheet';

import cx from './TravellersCount.scss';

export interface ITravellersCountProps
    extends IWithClassName,
        IWithQaAttributes {
    min: number;
    max: number;
    value: number;
    label?: string;
    hint?: string;
    /**
     * Подсказка, которая всплывает при достижении максимального количества пассажиров
     */
    maxCountHint?: ReactNode;
    onChange(value: number): void;
    disabled?: boolean;
    maxWidth?: boolean;
    /**
     * Расстояние между + и -
     */
    countWidth?: 34 | 50;
}

const TravellersCount: React.FC<ITravellersCountProps> = props => {
    const {
        className,
        onChange,
        value,
        min,
        max,
        label,
        hint,
        maxCountHint,
        disabled,
        maxWidth = true,
        countWidth = 50,
    } = props;

    const deviceType = useDeviceType();

    const minusDisabled = value <= min || disabled;
    const plusDisabled = value >= max || disabled;

    const plusButtonWrapperRef = useRef<HTMLDivElement | null>(null);

    const {
        handleMouseEnter: showHint,
        handleMouseLeave: hideHint,
        isHovered: isHintVisible,
    } = useTooltipHover(deviceType.isMobile ? 0 : undefined);

    const messageBoxProps = useMemo(
        () => ({
            anchorRef: plusButtonWrapperRef,
            hasClose: false,
        }),
        [],
    );

    const handleMinusClick = useCallback(() => {
        onChange(value - 1);
    }, [onChange, value]);

    const handlePlusClick = useCallback(() => {
        onChange(value + 1);
    }, [onChange, value]);

    return (
        <div
            className={cx(
                'travellersCount',
                {travellersCount_maxWidth: maxWidth},
                className,
            )}
        >
            {(label || hint) && (
                <div className={cx('label')}>
                    {label && <div className={cx('labelTitle')}>{label}</div>}
                    {hint && <div className={cx('labelHint')}>{hint}</div>}
                </div>
            )}

            <div className={cx('controls')}>
                <Button
                    size="l"
                    theme="secondary"
                    shape="circle"
                    onClick={handleMinusClick}
                    disabled={minusDisabled}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'minusButton',
                    })}
                >
                    <MinusIcon />
                </Button>
                <div
                    className={cx('count')}
                    style={{width: countWidth}}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'value',
                    })}
                >
                    {value}
                </div>
                <div
                    ref={plusButtonWrapperRef}
                    onClick={plusDisabled ? showHint : undefined}
                    onMouseEnter={
                        deviceType.isDesktop && plusDisabled
                            ? showHint
                            : undefined
                    }
                    onMouseLeave={
                        deviceType.isDesktop && plusDisabled
                            ? hideHint
                            : undefined
                    }
                >
                    <Button
                        size="l"
                        theme="secondary"
                        shape="circle"
                        onClick={handlePlusClick}
                        disabled={plusDisabled}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'plusButton',
                        })}
                    >
                        <PlusIcon />
                    </Button>
                </div>
            </div>

            <MessageBoxPopupOrBottomSheet
                isVisible={isHintVisible}
                messageBoxProps={messageBoxProps}
                onClose={hideHint}
            >
                <div
                    className={cx(
                        'maxCountHint',
                        deviceMods('maxCountHint', deviceType),
                    )}
                >
                    {maxCountHint}
                </div>
            </MessageBoxPopupOrBottomSheet>
        </div>
    );
};

export default TravellersCount;
