import {FunctionComponent, ReactNode, useRef} from 'react';

import {IWithClassName} from 'types/withClassName';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import MessageBoxPopup, {
    EMessageBoxPopupTheme,
} from 'components/MessageBoxPopup/MessageBoxPopup';

import ScopeContext from 'contexts/ScopeContext';

import TriggerButton from '../TriggerButton/TriggerButton';

import cx from './TriggerDetailsButtonDesktop.scss';

const DEFAULT_DIRECTIONS = [
    EPopupDirection.BOTTOM,
    EPopupDirection.BOTTOM_LEFT,
    EPopupDirection.BOTTOM_RIGHT,
    EPopupDirection.TOP,
];

export interface ITriggerDetailsButtonDesktopProps
    extends IWithClassName,
        IWithQaAttributes {
    iconClassName?: string;
    theme?: EMessageBoxPopupTheme;
}

const TriggerDetailsButtonDesktop: FunctionComponent<ITriggerDetailsButtonDesktopProps> =
    props => {
        const {
            className,
            iconClassName,
            children,
            theme = EMessageBoxPopupTheme.DEFAULT,
        } = props;
        const buttonRef = useRef(null);
        const {
            value: isVisible,
            setTrue: showPopup,
            setFalse: hidePopup,
        } = useBoolean(false);

        return (
            <>
                <TriggerButton
                    className={className}
                    iconClassName={iconClassName}
                    buttonRef={buttonRef}
                    onClick={showPopup}
                    {...prepareQaAttributes(props)}
                />
                <ScopeContext.Consumer>
                    {(scopeElement): ReactNode => (
                        <MessageBoxPopup
                            isVisible={isVisible}
                            direction={DEFAULT_DIRECTIONS}
                            anchorRef={buttonRef}
                            scopeRef={
                                scopeElement
                                    ? {current: scopeElement}
                                    : undefined
                            }
                            theme={theme}
                            onClose={hidePopup}
                            zIndexGroupLevel={1}
                        >
                            <div className={cx('tooltipContent')}>
                                {children}
                            </div>
                        </MessageBoxPopup>
                    )}
                </ScopeContext.Consumer>
            </>
        );
    };

export default TriggerDetailsButtonDesktop;
