import React, {useCallback, useEffect} from 'react';
import ReactTruncate from 'react-truncate';

import {IWithClassName} from 'types/withClassName';

import {CHAR_NBSP} from 'utilities/strings/charCodes';
import {useBoolean} from 'utilities/hooks/useBoolean';

import * as i18nBlock from 'i18nNew/common';

import LinkButton from 'components/LinkButton/LinkButton';

import cx from './TruncateText.scss';

interface ITruncateTextProps extends IWithClassName {
    text: string;
    expanderText?: string;
    linesCount?: number;
    expanderClassName?: string;
    collapsable?: boolean;
    collapserText?: string;
    collapserClassName?: string;
}

const DEFAULT_LINES_COUNT = 3;

const TruncateText: React.FC<ITruncateTextProps> = ({
    text,
    className,
    linesCount = DEFAULT_LINES_COUNT,
    expanderText = i18nBlock.readDashMore(),
    expanderClassName,
    collapsable,
    collapserText = i18nBlock.less(),
    collapserClassName,
}) => {
    const {
        value: canRenderAllLine,
        setFalse: collapseText,
        toggle: toggleText,
    } = useBoolean(false);

    const handleExpand = useCallback(
        (event: React.MouseEvent) => {
            // Если текст располагается внутри элемента с которым можно взаимодействовать,
            // то разворачивание текста не должно триггерить основное действие
            event.stopPropagation();
            toggleText();
        },
        [toggleText],
    );

    useEffect(() => {
        collapseText();
    }, [text, collapseText]);

    return (
        <ReactTruncate
            className={className}
            lines={canRenderAllLine ? false : linesCount}
            ellipsis={
                <span>
                    {`...${CHAR_NBSP}`}
                    <LinkButton
                        className={expanderClassName}
                        onClick={handleExpand}
                    >
                        {expanderText}
                    </LinkButton>
                </span>
            }
        >
            {text}
            {collapsable && canRenderAllLine && (
                <LinkButton
                    className={cx('collapser', collapserClassName)}
                    onClick={handleExpand}
                >
                    {collapserText}
                </LinkButton>
            )}
        </ReactTruncate>
    );
};

export default TruncateText;
