import React, {useCallback, useMemo} from 'react';

import {MINUTE} from 'utilities/dateUtils/constants';

import {reloadPage} from 'utilities/browser/reloadPage';

import * as i18nBlock from 'i18nNew/components-UpdateNotification';

import ErrorModal, {TErrorActionType} from 'components/ErrorModal/ErrorModal';

import TicketImg from './icons/ticket.svg';
import useUpdateNotification from './hooks/useUpdateNotification';

export interface IUpdateNotificationProps {
    /**
     * Текст заголовка
     */
    title?: string;

    /**
     * Текст сообщения
     */
    text?: string;

    /**
     * Текст в кнопке для обновления
     */
    primaryActionText?: string;

    /**
     * Текст в кнопке для закрытия попапа без обновления
     */
    secondaryActionText?: string;

    /**
     * Время в миллисекундах, через которое появляется уведомление
     */
    timeout?: number;

    /**
     * Принудительно не показывать модал,
     * но все таймеры работают и при снятии этого флага модал будет показан
     */
    forceHide?: boolean;

    /**
     * Обработчик на кнопку для обновления
     */
    onUpdate?(): void;

    /**
     * Callback на показ уведомления
     */
    onShowNotification?(): void;

    /**
     * Указывает, нужно ли перезагружать страницу после нажатия на кнопку
     */
    canReloadPageAfterAction?: boolean;

    /**
     * Указывает, можно ли закрыть окно без перезагрузки страницы
     */
    isSkippable?: boolean;

    /**
     * Callback на закрытие без экшна
     */
    onSkip?(): void;
}

const UpdateNotification: React.FC<IUpdateNotificationProps> = ({
    title = i18nBlock.title(),
    text = i18nBlock.text(),
    primaryActionText = i18nBlock.actionText(),
    secondaryActionText = i18nBlock.secondaryActionText(),
    timeout = 15 * MINUTE,
    forceHide = false,
    canReloadPageAfterAction = true,
    onUpdate,
    onShowNotification,
    isSkippable = true,
    onSkip,
}) => {
    const {isVisible, close} = useUpdateNotification({
        timeout,
        onShowNotification,
    });

    const handlePrimaryAction = useCallback(() => {
        onUpdate?.();

        if (canReloadPageAfterAction) {
            reloadPage();
        }
    }, [onUpdate, canReloadPageAfterAction]);

    const handleSecondaryAction = useCallback(() => {
        onSkip?.();
        close();
    }, [close, onSkip]);

    const primaryAction = useMemo(
        (): TErrorActionType => ({
            type: 'button',
            title: primaryActionText,
            handler: handlePrimaryAction,
        }),
        [handlePrimaryAction, primaryActionText],
    );

    const secondaryAction = useMemo(
        (): TErrorActionType => ({
            type: 'button',
            title: secondaryActionText,
            handler: handleSecondaryAction,
        }),
        [handleSecondaryAction, secondaryActionText],
    );

    const image = useMemo(() => {
        return <img src={TicketImg} alt="ticket" />;
    }, []);

    return (
        <ErrorModal
            title={title}
            text={text}
            isVisible={forceHide ? false : isVisible}
            image={image}
            primaryAction={primaryAction}
            secondaryAction={isSkippable ? secondaryAction : undefined}
            isClosable={isSkippable}
            onClose={handleSecondaryAction}
        />
    );
};

export default UpdateNotification;
