import React, {useEffect, useRef} from 'react';

import {IWithClassName} from 'types/withClassName';
import {EUserAvatarSize} from 'components/User/types';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {TUserInfo} from 'reducers/common/commonReducerTypes';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {loadable} from 'utilities/componentLoadable';
import usePreloadChunks from 'utilities/hooks/usePreloadChunks';
import {EPassportMode, getPassportUrl} from 'utilities/url/getPassportUrl';
import useSupportHistoryBack from 'utilities/hooks/useSupportHistoryBack';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {useEventListener} from 'utilities/hooks/useEventListener';

import * as i18nBlock from 'i18nNew/yandexUser';

import Flex from 'components/Flex/Flex';
import Popup from 'components/Popup/Popup';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import AvatarAndLogin from 'components/User/components/AvatarAndLogin/AvatarAndLogin';
import useReturnPathUrl from 'components/User/hooks/useReturnPathUrl';

import cx from './User.scss';

interface IUserProps extends IWithClassName, IWithQaAttributes {
    userInfo: TUserInfo;
    unreadMailCount: number;
    avatarSize?: EUserAvatarSize;
    withName?: boolean;
    fetchUserUnreadMailCount(): void;
}

const PopupContentLoadable = loadable(
    () => import('components/User/components/PopupContent/PopupContent'),
    {
        hasFallback: true,
        spinnerSize: 'm',
        className: cx('loadablePopup'),
    },
);

const CHUNKS_TO_PRELOAD = [PopupContentLoadable];

const POPUP_DIRECTIONS = [EPopupDirection.BOTTOM_RIGHT];

const User: React.FC<IUserProps> = props => {
    const {
        className,
        userInfo,
        unreadMailCount,
        avatarSize = EUserAvatarSize.M,
        withName = false,
        fetchUserUnreadMailCount,
    } = props;

    const wrapperRef = useRef<HTMLDivElement | null>(null);

    const {
        value: popupIsVisible,
        setTrue: showPopup,
        setFalse: hidePopup,
    } = useBoolean(false);

    const returnPathUrl = useReturnPathUrl();

    usePreloadChunks(CHUNKS_TO_PRELOAD);

    useSupportHistoryBack({
        isVisible: popupIsVisible,
        onHistoryBack: hidePopup,
        close: hidePopup,
    });

    useEffect(() => {
        fetchUserUnreadMailCount();
    }, [fetchUserUnreadMailCount]);

    useEventListener('scroll', hidePopup);

    if (isAuthUser(userInfo)) {
        return (
            <div className={className} ref={wrapperRef}>
                <Flex between={3} inline>
                    <AvatarAndLogin
                        className={cx('authAvatarToggle')}
                        loginClassName={cx('login')}
                        avatarPath={userInfo.avatarPath}
                        avatarId={userInfo.avatar.default}
                        login={withName ? userInfo.displayName : undefined}
                        loginFirstLetterRed
                        size={avatarSize}
                        hasPlus={userInfo.havePlus}
                        unreadMailCount={unreadMailCount}
                        onClick={showPopup}
                    />
                </Flex>

                <Popup
                    className={cx('popup', `popup_shift_${avatarSize}`)}
                    anchor={wrapperRef}
                    visible={popupIsVisible}
                    directions={POPUP_DIRECTIONS}
                    zIndex={100000}
                    onClose={hidePopup}
                >
                    <PopupContentLoadable
                        userInfo={userInfo}
                        unreadMailCount={unreadMailCount}
                        returnPathUrl={returnPathUrl}
                        hidePopup={hidePopup}
                    />
                </Popup>
            </div>
        );
    }

    return (
        <ButtonLink
            className={className}
            theme="secondary"
            size="m"
            url={getPassportUrl({
                mode: EPassportMode.AUTH,
                passportHost: userInfo.passportPath,
                retpath: returnPathUrl,
            })}
            {...prepareQaAttributes({
                parent: props,
                current: 'loginButton',
            })}
        >
            {i18nBlock.login()}
        </ButtonLink>
    );
};

export default React.memo(User);
