import React from 'react';

import {IWithClassName} from 'types/withClassName';
import {EUserAvatarSize} from 'components/User/types';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {EPassportMode, getPassportUrl} from 'utilities/url/getPassportUrl';
import getAvatarUrl from 'utilities/avatar/getAvatarUrl';

import CameraIcon from 'icons/36/Camera';
import AvatarYandexPlusIconM from 'icons/illustrations/AvatarYandexPlusIcon/AvatarYandexPlusIconM';

import cx from './Avatar.scss';

export interface IChangePhotoOptions {
    visible: boolean;
    passportPath: string;
    returnPathUrl: string;
}

interface IAvatarProps extends IWithClassName {
    avatarPath: string;
    avatarId: string;
    size: EUserAvatarSize;
    changePhotoOptions?: IChangePhotoOptions;
    hasPlus?: boolean;
    unreadMailCount?: number;
}

const Avatar: React.FC<IAvatarProps> = props => {
    const {
        className,
        avatarPath,
        avatarId,
        size = EUserAvatarSize.M,
        changePhotoOptions,
        hasPlus = false,
        unreadMailCount,
    } = props;

    const deviceType = useDeviceType();

    const commonSrc = getAvatarUrl(avatarPath, avatarId);
    const retinaSrc = getAvatarUrl(avatarPath, avatarId, {retina: true});

    return (
        <div
            className={cx(
                'root',
                `root_size_${size}`,
                {root_hasPlus: hasPlus},
                deviceMods('root', deviceType),
                className,
            )}
        >
            <img
                className={cx('image')}
                src={commonSrc}
                srcSet={`${commonSrc} 1x, ${retinaSrc} 2x`}
            />

            {changePhotoOptions?.visible && (
                <a
                    className={cx('changePhotoWrapper')}
                    href={getPassportUrl({
                        mode: EPassportMode.PROFILE,
                        passportHost: changePhotoOptions.passportPath,
                        retpath: changePhotoOptions.returnPathUrl,
                    })}
                >
                    <div className={cx('changePhoto')}>
                        <CameraIcon className={cx('changePhotoIcon')} />
                    </div>
                </a>
            )}

            {hasPlus && <AvatarYandexPlusIconM className={cx('plus')} />}

            {unreadMailCount ? (
                <div className={cx('unreadMailCount')}>{unreadMailCount}</div>
            ) : null}
        </div>
    );
};

export default React.memo(Avatar);
