import React, {useMemo} from 'react';

import {EUserAvatarSize} from 'components/User/types';

import {IAuthUserInfo} from 'reducers/common/commonReducerTypes';

import {EPassportMode, getPassportUrl} from 'utilities/url/getPassportUrl';

import * as i18nBlock from 'i18nNew/yandexUser';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Link from 'components/Link/Link';
import AvatarAndLogin from 'components/User/components/AvatarAndLogin/AvatarAndLogin';
import YandexPlusM from 'icons/illustrations/YandexPlus/YandexPlusM';
import {IChangePhotoOptions} from 'components/User/components/AvatarAndLogin/components/Avatar/Avatar';

import cx from './Menu.scss';

interface IMenuProps {
    userInfo: IAuthUserInfo;
    unreadMailCount: number;
    returnPathUrl: string;
}

const Menu: React.FC<IMenuProps> = props => {
    const {userInfo, unreadMailCount, returnPathUrl} = props;

    const changePhotoOptions = useMemo<IChangePhotoOptions>(() => {
        return {
            visible: userInfo.avatar.empty,
            passportPath: userInfo.passportPath,
            returnPathUrl,
        };
    }, [returnPathUrl, userInfo.avatar.empty, userInfo.passportPath]);

    return (
        <Box x={5} y={4} between={1}>
            <AvatarAndLogin
                avatarPath={userInfo.avatarPath}
                avatarId={userInfo.avatar.default}
                login={userInfo.displayName}
                size={EUserAvatarSize.L}
                changePhotoOptions={changePhotoOptions}
            />

            <Flex className={cx('menu')} flexDirection="column">
                <Link
                    className={cx('menuItem')}
                    url="https://mail.yandex.ru/"
                    theme="black"
                >
                    {i18nBlock.mail()}

                    {unreadMailCount ? (
                        <span className={cx('unreadMailCount')}>
                            {unreadMailCount}
                        </span>
                    ) : null}
                </Link>

                <Link
                    className={cx(
                        'menuItem',
                        userInfo.havePlus ? '' : 'getPlus',
                    )}
                    url={'YANDEX_PLUS_URL'}
                    theme={userInfo.havePlus ? 'black' : 'normal'}
                >
                    {userInfo.havePlus ? (
                        <Flex alignItems="center" between={1} inline>
                            <YandexPlusM width={40} height={16} />

                            <span>{i18nBlock.activeYandexPlus()}</span>
                        </Flex>
                    ) : (
                        i18nBlock.getYandexPlus()
                    )}
                </Link>

                <Link
                    className={cx('menuItem', 'manageAccount')}
                    url={userInfo.passportPath}
                    theme="black"
                >
                    {i18nBlock.manageAccount()}
                </Link>

                <Link
                    className={cx('menuItem')}
                    url={getPassportUrl({
                        mode: EPassportMode.LOGOUT,
                        passportHost: userInfo.passportPath,
                        yu: userInfo.yu,
                        uid: userInfo.uid,
                        retpath: returnPathUrl,
                    })}
                    theme="black"
                >
                    {i18nBlock.logout()}
                </Link>
            </Flex>
        </Box>
    );
};

export default React.memo(Menu);
