import React, {Component} from 'react';
import {
    Polyline as YandexMapPolyline,
    Circle as YandexMapCircle,
    YMaps as YandexMaps,
    Map as YandexMap,
    Placemark as YandexMapPlacemark,
    ZoomControl,
    GeoObject,
    IYMapsApi,
    IYMapsInstance,
    IMapState,
    ITickEvent,
    IYaBoundsChangeEvent,
    YaMapCoordinatesType,
    IUserGeoLocationEvent,
} from 'react-yandex-maps';
import _flow from 'lodash/flow';

import {ICoordinates, MapBoundsType} from 'types/common/ICoordinates';
import {IWithClassName} from 'types/withClassName';

import {
    convertBoundsToMapAPI,
    convertCoordinatesToMapAPI,
    convertCoordinates,
    convertBounds,
    isDifferenceBounds,
} from './utilities';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Spinner from 'components/Spinner/Spinner';
import PlusIcon from 'icons/16/Plus';
import MinusIcon from 'icons/16/Minus';
import GeolocationIcon from 'icons/16/Geolocation';
import Button from 'components/Button/Button';
import Flex from 'components/Flex/Flex';

import {useYandexMapInstance} from './hooks/useYandexMapInstance';

import cx from './YandexMaps.scss';

const YANDEX_MAP_DEFAULT_CENTER: ICoordinates = {
    lat: 55.75,
    lon: 37.57,
};
const YANDEX_MAP_DEFAULT_ZOOM = 13;
const YANDEX_MAP_MAX_ZOOM = 19;
const YANDEX_MAP_MIN_ZOOM = 5;
const YANDEX_MAP_USER_GEO_ZOOM = 15;

export const YMAPS_JS_API_KEY = 'cac18eae-4d25-4feb-9916-dc5e433a0668';

interface IYandexMapsState {
    isLoading: boolean;
    zoom?: number;
    center?: ICoordinates;
    userPosition: ICoordinates | null;
    isLoadingUserGeoPosition: boolean;
}

export interface IYandexMapsProps extends IWithQaAttributes, IWithClassName {
    customControlsClassName?: string;
    mapStyle?: React.CSSProperties;
    defaultCenter?: ICoordinates;
    defaultBounds?: MapBoundsType;
    defaultZoom?: number;
    defaultMargins?: number[] | number[][];
    yandexMapInstanceRef?: (map: IYMapsInstance) => void;
    providerQuery?: {
        ns: string;
        load: string;
    };
    minZoom?: number;
    maxZoom?: number;
    modules?: string[];
    yandexMapDisablePoiInteractivity?: boolean;
    avoidFractionalZoom?: boolean;
    hasGeoLocationControl?: boolean;
    hasZoomControl?: boolean;
    controls?: string[];
    mapMarkers?: ICoordinates[];
    children?: React.ReactNode;
    /* Handlers */
    onLoadMap?: (map: IYMapsApi) => void;
    onActionBegin?: () => void;
    onActionEnd?: () => void;
    onStartChangeZoom?: (zoom: number) => void;
    onBoundsChange?: (
        bounds: MapBoundsType,
        {isUserEvent}: {isUserEvent: boolean},
    ) => void;
    onClickUpZoomButton?: () => void;
    onClickDownZoomButton?: () => void;
    onClick?: () => void;
}

class YandexMapsComponent extends Component<
    IYandexMapsProps,
    IYandexMapsState
> {
    readonly state: IYandexMapsState = {
        isLoading: true,
        isLoadingUserGeoPosition: false,
        userPosition: null,
    };

    private _preventRender: boolean = false;

    private _startSyncMapBoundsByProps: boolean = false;

    private yandexMapAPIInstance: IYMapsApi = {};

    private yandexMapInstance: IYMapsInstance = {};

    static readonly defaultProps = {
        defaultCenter: YANDEX_MAP_DEFAULT_CENTER,
        defaultZoom: YANDEX_MAP_DEFAULT_ZOOM,
        providerQuery: {
            ns: 'use-load-option',
            load: 'package.full',
            apikey: YMAPS_JS_API_KEY,
        },
        modules: [],
        controls: [],
        minZoom: YANDEX_MAP_MIN_ZOOM,
        maxZoom: YANDEX_MAP_MAX_ZOOM,
        mapMarkers: [],
        children: null,
        hasGeoLocationControl: false,
        hasZoomControl: true,
    };

    shouldComponentUpdate(
        _nextProps: IYandexMapsProps,
        nextState: IYandexMapsState,
    ): boolean {
        if (this.state.isLoading !== nextState.isLoading) {
            return true;
        }

        return !this._preventRender;
    }

    componentDidUpdate(prevProps: IYandexMapsProps): void {
        this.checkUpdateBounds(prevProps);
        this.checkUpdateCenterAndZoom(prevProps);
    }

    /* Update Map by changed props [defaultCenter, defaultZoom, defaultBounds] */

    private checkUpdateBounds(prevProps: IYandexMapsProps): void {
        const {defaultBounds: prevDefaultBounds} = prevProps;
        const {defaultBounds: nextDefaultBounds} = this.props;

        if (
            nextDefaultBounds &&
            isDifferenceBounds(prevDefaultBounds, nextDefaultBounds)
        ) {
            this.setMapBounds(nextDefaultBounds);
        }
    }

    private checkUpdateCenterAndZoom(prevProps: IYandexMapsProps): void {
        const {defaultCenter: prevDefaultCenter, defaultZoom: prevDefaultZoom} =
            prevProps;
        const {defaultCenter: nextDefaultCenter, defaultZoom: nextDefaultZoom} =
            this.props;

        if (
            nextDefaultCenter !== prevDefaultCenter ||
            nextDefaultZoom !== prevDefaultZoom
        ) {
            this.setState({
                zoom: nextDefaultZoom,
                center: nextDefaultCenter,
            });
        }
    }

    /* Prepare defaultMapAPIState by defaultCenter, defaultZoom, defaultBounds from props */

    private getDefaultMapState = (): IMapState => {
        const {
            defaultCenter,
            defaultZoom,
            defaultBounds,
            defaultMargins,
            controls,
        } = this.props;

        if (defaultBounds) {
            return {
                bounds: convertBoundsToMapAPI(defaultBounds),
                margin: defaultMargins,
                controls,
            };
        }

        return {
            center: defaultCenter && convertCoordinatesToMapAPI(defaultCenter),
            zoom: defaultZoom,
            margin: defaultMargins,
            controls,
        };
    };

    /* Prepare mapAPIState by component state */

    private getMapState = (): IMapState | undefined => {
        const {controls} = this.props;
        const {center, zoom} = this.state;

        if (center && zoom) {
            return {
                center: convertCoordinatesToMapAPI(center),
                zoom,
                controls,
            };
        }
    };

    /* Setters and getters */

    private setYandexMapInstance = (
        yandexMapInstance: IYMapsInstance,
    ): void => {
        const {yandexMapInstanceRef} = this.props;

        this.yandexMapInstance = yandexMapInstance;

        if (yandexMapInstanceRef) {
            yandexMapInstanceRef(yandexMapInstance);
        }
    };

    private setYandexMapAPIInstance = (
        yandexMapAPIInstance: IYMapsApi,
    ): void => {
        this.yandexMapAPIInstance = yandexMapAPIInstance;
    };

    /* Map success loaded */

    private fillComponentState = (): void => {
        if (this.yandexMapInstance) {
            const zoom: number = this.yandexMapInstance.getZoom();
            const center: YaMapCoordinatesType =
                this.yandexMapInstance.getCenter();

            this.setState({
                zoom,
                center: convertCoordinates(center),
                isLoading: false,
            });
        }
    };

    private handleLoadMap = (yandexMapAPIInstance: IYMapsApi): void => {
        const {onLoadMap} = this.props;

        this.setYandexMapAPIInstance(yandexMapAPIInstance);

        if (onLoadMap) {
            onLoadMap(yandexMapAPIInstance);
        }

        this.fillComponentState();
    };

    /* Map bounds changed */

    private handleActionBegin = (): void => {
        const {onActionBegin} = this.props;

        this._preventRender = true;

        if (onActionBegin) {
            onActionBegin();
        }
    };

    private handleActionTick = (tickEvent: ITickEvent): void => {
        const {onStartChangeZoom} = this.props;
        const {
            originalEvent: {
                tick: {zoom: newZoom},
            },
        } = tickEvent;
        const {zoom: currentZoom} = this.state;

        if (newZoom !== currentZoom && onStartChangeZoom) {
            onStartChangeZoom(newZoom);
        }
    };

    private handleActionEnd = (): void => {
        const {onActionEnd} = this.props;

        this._preventRender = false;

        if (onActionEnd) {
            onActionEnd();
        }
    };

    private handleBoundsChange = (boundsProps: IYaBoundsChangeEvent): void => {
        const {onBoundsChange} = this.props;
        const {
            originalEvent: {newZoom, newCenter, newBounds},
        } = boundsProps;

        this._preventRender = false;

        this.setState({
            center: convertCoordinates(newCenter),
            zoom: newZoom,
        });

        if (onBoundsChange) {
            onBoundsChange(convertBounds(newBounds), {
                isUserEvent: !this._startSyncMapBoundsByProps,
            });
        }
    };

    /* Zoom control handlers */

    private handleClickUpZoomButton = (): void => {
        const {zoom} = this.state;
        const {onClickUpZoomButton} = this.props;

        if (zoom) {
            this.setMapZoom(zoom + 1);
        }

        if (onClickUpZoomButton) {
            onClickUpZoomButton();
        }
    };

    private handleClickDownZoomButton = (): void => {
        const {zoom} = this.state;
        const {onClickDownZoomButton} = this.props;

        if (zoom) {
            this.setMapZoom(zoom - 1);
        }

        if (onClickDownZoomButton) {
            onClickDownZoomButton();
        }
    };

    /* UserLocation controls handlers */

    private handleLoadUserGeoLocation = (
        userGeoLocation: IUserGeoLocationEvent,
    ): void => {
        const {
            geoObjects: {position},
        } = userGeoLocation;

        const userPosition = convertCoordinates(position);

        this.setState({
            userPosition,
            center: userPosition,
            zoom: YANDEX_MAP_USER_GEO_ZOOM,
            isLoadingUserGeoPosition: false,
        });
    };

    private handleClickUserGeoLocation = (): void => {
        const {isLoadingUserGeoPosition} = this.state;

        if (isLoadingUserGeoPosition) {
            return;
        }

        this.setState({isLoadingUserGeoPosition: true});

        this.yandexMapAPIInstance.geolocation
            .get({mapStateAutoApply: true})
            .then(this.handleLoadUserGeoLocation);
    };

    private setMapZoom(nextZoom: number): void {
        const {minZoom = YANDEX_MAP_MIN_ZOOM, maxZoom = YANDEX_MAP_MAX_ZOOM} =
            this.props;

        if (Number.isFinite(nextZoom)) {
            const zoom = _flow(
                (zoomValue: number) => Math.min(zoomValue, maxZoom),
                (zoomValue: number) => Math.max(zoomValue, minZoom),
            )(nextZoom);

            this.setState({zoom});
        }
    }

    private async setMapBounds(nextBounds: MapBoundsType): Promise<void> {
        if (this.yandexMapInstance && this.yandexMapInstance.setBounds) {
            const mapAPIBounds = this.yandexMapInstance.getBounds();
            const mapBounds = convertBounds(mapAPIBounds);

            if (isDifferenceBounds(mapBounds, nextBounds)) {
                this._startSyncMapBoundsByProps = true;

                await this.yandexMapInstance.setBounds(
                    convertBoundsToMapAPI(nextBounds),
                );

                this._startSyncMapBoundsByProps = false;
            }
        }
    }

    /* Render */

    private renderGeoLocationControl(): React.ReactNode {
        const {isLoadingUserGeoPosition} = this.state;

        return (
            <Button
                theme="raised"
                shape="circle"
                onClick={this.handleClickUserGeoLocation}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'currentGeoLocation',
                })}
            >
                {isLoadingUserGeoPosition ? (
                    <Spinner size="xs" />
                ) : (
                    <GeolocationIcon />
                )}
            </Button>
        );
    }

    private renderUserGeoPositionMarker(): React.ReactNode {
        const {userPosition} = this.state;

        if (userPosition) {
            return (
                <YandexMapPlacemark
                    defaultGeometry={convertCoordinatesToMapAPI(userPosition)}
                    options={{
                        preset: 'islands#geolocationIcon',
                    }}
                />
            );
        }

        return null;
    }

    private renderCustomControls(): React.ReactNode {
        const {hasGeoLocationControl, hasZoomControl, customControlsClassName} =
            this.props;

        return (
            <Flex
                className={cx('customControls', customControlsClassName)}
                flexDirection="column"
                between={2}
            >
                {hasZoomControl && this.renderCustomZoomControl()}
                {hasGeoLocationControl && this.renderGeoLocationControl()}
            </Flex>
        );
    }

    private renderCustomZoomControl(): React.ReactNode {
        const {zoom} = this.state;
        const {minZoom, maxZoom} = this.props;

        return (
            <>
                <Button
                    theme="raised"
                    shape="circle"
                    disabled={zoom === maxZoom}
                    onClick={this.handleClickUpZoomButton}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'zoomIn',
                    })}
                >
                    <PlusIcon />
                </Button>

                <Button
                    theme="raised"
                    shape="circle"
                    disabled={zoom === minZoom}
                    onClick={this.handleClickDownZoomButton}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'zoomOut',
                    })}
                >
                    <MinusIcon />
                </Button>
            </>
        );
    }

    private renderMarkers(): React.ReactNode {
        const {mapMarkers} = this.props;

        return mapMarkers?.map(({lat, lon}) => (
            <YandexMapPlacemark
                key={`${lat}-${lon}`}
                defaultGeometry={[lat, lon]}
            />
        ));
    }

    render(): React.ReactNode {
        const {
            children,
            minZoom,
            maxZoom,
            className,
            mapStyle,
            modules,
            avoidFractionalZoom,
            yandexMapDisablePoiInteractivity,
            providerQuery,
            onClick,
        } = this.props;
        const {isLoading} = this.state;

        return (
            <YandexMaps query={providerQuery}>
                {isLoading && (
                    <div className={cx('loader')}>
                        <Spinner className={cx('spinner')} />
                    </div>
                )}
                <YandexMap
                    instanceRef={this.setYandexMapInstance}
                    onLoad={this.handleLoadMap}
                    onActionBegin={this.handleActionBegin}
                    onActionTick={this.handleActionTick}
                    onActionEnd={this.handleActionEnd}
                    onBoundsChange={this.handleBoundsChange}
                    onClick={onClick}
                    className={cx('map', className)}
                    style={mapStyle}
                    options={{
                        minZoom,
                        maxZoom,
                        avoidFractionalZoom,
                        yandexMapDisablePoiInteractivity,
                    }}
                    defaultState={this.getDefaultMapState()}
                    state={this.getMapState()}
                    modules={modules}
                >
                    {this.renderCustomControls()}
                    {this.renderUserGeoPositionMarker()}
                    {this.renderMarkers()}
                    {children}
                </YandexMap>
            </YandexMaps>
        );
    }
}

export {
    YandexMaps,
    YandexMap,
    YandexMapCircle,
    YandexMapPolyline,
    YandexMapPlacemark,
    ZoomControl,
    GeoObject,
    useYandexMapInstance,
};

export default YandexMapsComponent;
