import {
    FunctionComponent,
    useCallback,
    useMemo,
    useRef,
    useEffect,
    useState,
} from 'react';
import {useSelector} from 'react-redux';
import {
    YandexConfig,
    Widget,
    popupUIFactory,
    OnToggledEvent,
} from '@yandex-int/messenger.widget';

import {
    TRIGGER_NODE_ID,
    YANDEX_TRAVEL_BOT_ID,
    YANDEX_TRAVEL_SERVICE_ID,
    YANDEX_TRAVEL_WORKSPACE_ID,
} from './constants';

import {IWithClassName} from 'types/withClassName';
import {TMetrikaGoal} from 'utilities/metrika/types/goals/all';

import {getUserInfo} from 'selectors/common/userInfoSelector';
import appVersionSelector from 'selectors/common/appVersion';

import {prepareMetadata, TEntryPoint} from './utilities/prepareMetadata';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {reachGoal} from 'utilities/metrika';

import Flex from 'components/Flex/Flex';
import Text, {TTextSize} from 'components/Text/Text';
import Overlay from 'components/Overlay/Overlay';
import LinkButton from 'components/LinkButton/LinkButton';
import MessageOutlineIcon from 'icons/24/MessageOutline';

if (__CLIENT__) {
    require('@yandex-int/messenger.widget/lib/ui/popup.css');
}

import cx from './YandexMessenger.scss';

interface IYandexMessengerProps extends IWithClassName {
    label?: string;
    labelSize?: TTextSize;
    title?: string;
    entrypoint: TEntryPoint;
    metrikaGoal: TMetrikaGoal;
}

const YandexMessenger: FunctionComponent<IYandexMessengerProps> = ({
    className,
    title,
    label,
    labelSize,
    entrypoint,
    metrikaGoal,
}) => {
    const userInfo = useSelector(getUserInfo);
    const appVersion = useSelector(appVersionSelector);
    const deviceType = useDeviceType();
    const chatWidget = useRef<Widget | null>(null);
    const triggerNodeId = useMemo(
        () => `${TRIGGER_NODE_ID}_${entrypoint}`,
        [entrypoint],
    );
    const [isOverlay, setOverlay] = useState(false);

    const chatMetadata = useMemo(
        () =>
            prepareMetadata({
                userInfo,
                deviceType,
                appVersion,
                entrypoint,
            }),
        [userInfo, deviceType, appVersion, entrypoint],
    );

    const handleGetEnv = useCallback(
        request => {
            try {
                request.response(chatMetadata);
            } catch (error) {
                request.reject(error);
            }
        },
        [chatMetadata],
    );

    const handleChatToggle = useCallback(
        (e: OnToggledEvent) => {
            setOverlay(!e.hidden);
        },
        [setOverlay],
    );

    const handleClick = useCallback(() => {
        if (deviceType.isDesktop) {
            setOverlay(true);
        }

        chatWidget.current?.show();
        reachGoal(metrikaGoal);
    }, [chatWidget, deviceType, setOverlay, metrikaGoal]);

    const handleOverlayClick = useCallback(() => {
        setOverlay(false);
        chatWidget.current?.hide();
    }, [chatWidget, setOverlay]);

    useEffect(() => {
        const ui = popupUIFactory({
            popupTargetNode:
                document.getElementById(triggerNodeId) || undefined,
        });

        const widget = new Widget(
            new YandexConfig({
                serviceId: YANDEX_TRAVEL_SERVICE_ID,
                workspaceId: YANDEX_TRAVEL_WORKSPACE_ID,
                iframeOpenData: {
                    guid: YANDEX_TRAVEL_BOT_ID,
                },
            }),
        )
            .setUI(ui)
            .init();

        widget.handlers.getEnv.addListener(handleGetEnv);

        ui.onToggled.addListener(handleChatToggle);
        ui.mount();

        chatWidget.current = widget;

        return (): void => {
            ui.onToggled.addListener(handleChatToggle);

            widget.handlers.getEnv.removeListener(handleGetEnv);
            widget.destroy();
        };
    }, [triggerNodeId, handleGetEnv, handleChatToggle]);

    return (
        <LinkButton className={className} theme="black" title={title}>
            <Flex
                tagProps={{id: triggerNodeId}}
                alignItems="center"
                between={2}
                inline
                onClick={handleClick}
            >
                <MessageOutlineIcon />
                {label && (
                    <Text
                        className={cx('text')}
                        color="current"
                        size={labelSize}
                    >
                        {label}
                    </Text>
                )}
            </Flex>
            <Overlay isVisible={isOverlay} onClick={handleOverlayClick} />
        </LinkButton>
    );
};

export default YandexMessenger;
