import React, {PureComponent} from 'react';
import {stringify as qsStringify, ParsedUrlQuery} from 'querystring';

import {IWithClassName} from 'types/withClassName';
import {ICoordinates} from 'types/common/ICoordinates';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18nNew/components-YandexStaticMaps';

const YMAPS_STATIC_API_KEY = '56d0c4b5-bdd4-4a28-9772-d90e4d88708d';

const YMAPS_STATIC_API = 'https://static-maps.yandex.ru/1.x/';

type TLayer = 'map' | 'sat' | 'skl' | 'trf';
/**
 * Компонент для использования Y.Maps Static API
 * @see https://tech.yandex.ru/maps/staticapi/doc/1.x/dg/concepts/input_params-docpage/
 **/
type TYandexStaticMapsProps = IWithClassName &
    IWithQaAttributes & {
        /**
         * Атрибут `<img>` width
         **/
        width: number;
        /**
         * Атрибут `<img>` height
         **/
        height: number;
        /**
         * Нужно ли использовать коммерческий ключ для Static Api
         * (Ключ работает для *.yandex.TLD)
         * @default true
         **/
        shouldUseKey?: boolean;
        /**
         * Перечень слоев, определяющих тип карты: map (схема), sat (спутник) и sat,skl (гибрид).
         **/
        layer: TLayer | TLayer[];
        coordinates: ICoordinates;
        /**
         * Уровень масштабирования карты (0-17)
         **/
        zoom?: number;
        /**
         * Ширина и высота запрашиваемого изображения карты (в пикселах)
         * Максимально допустимый размер изображения карты составляет 650x450 пикселов.
         **/
        size?: number[];
        /**
         * Коэффициент увеличения объектов на карте (от 1.0 до 4.0)
         * Пример: scale=2.0
         **/
        scale?: string;
        /**
         * Метки, которые требуется отобразить на карте.
         * (долготу, широту, стиль, цвет, размер и текст метки).
         * Описания меток перечисляются через тильду (~).
         * @see https://tech.yandex.ru/maps/staticapi/doc/1.x/dg/concepts/markers-docpage/
         * Пример: pt=37.620070,55.753630,pm2rdl
         **/
        pt?: string[];
        /**
         * Набор описаний геометрических фигур (ломаных и многоугольников),
         * которые требуется отобразить на карте.
         **/
        pl?: string;
        /**
         * Язык
         * Пример: lang=ru_RU
         **/
        lang?: string;
    };

class YandexStaticMaps extends PureComponent<TYandexStaticMapsProps> {
    static readonly defaultProps = {
        lang: 'ru_RU',
        pl: '',
        pt: [],
        scale: '1.0',
        size: [320, 320],
        zoom: 15,
        shouldUseKey: true,
    };

    private prepareQuery = (): ParsedUrlQuery => {
        const {
            layer,
            lang,
            size,
            zoom,
            scale,
            pt,
            pl,
            shouldUseKey,
            coordinates,
        } = this.props;

        return {
            l: Array.isArray(layer)
                ? layer.map(String).join(',')
                : String(layer),
            lang: String(lang),
            ll: [coordinates.lon, coordinates.lat].map(String).join(','), // Долгота и широта центра карты в градусах (LongitudeLatitude)
            pl: String(pl),
            pt: pt!.map(String).join('~'), // TS не понимает defaultProps
            scale: String(scale),
            size: size!.map(String).join(','), // TS не понимает defaultProps
            z: String(zoom),
            key: shouldUseKey ? YMAPS_STATIC_API_KEY : '',
        };
    };

    private getSrc(): string {
        const queryString = qsStringify(this.prepareQuery());

        return `${YMAPS_STATIC_API}?${queryString}`;
    }

    render(): React.ReactNode {
        const {className, width, height} = this.props;

        return (
            <div className={className} {...prepareQaAttributes(this.props)}>
                <img
                    src={this.getSrc()}
                    alt={i18nBlock.map()}
                    width={width}
                    height={height}
                />
            </div>
        );
    }
}

export default YandexStaticMaps;
