import {PagePoint} from '../../types';

import {pageToPoint} from '../../utilities';

/* global __CLIENT__ Hammer */
import AbstractZoomable from '../AbstractZoomable/AbstractZoomable';

if (__CLIENT__) {
    require('hammerjs');
}

export default class ZoomableMobile extends AbstractZoomable {
    private hammer?: HammerManager;

    private prevScale = 1;
    private disableClick = false;

    protected maxScale = 1;

    protected addListeners() {
        if (this.zoomableNode) {
            this.hammer = new Hammer(this.zoomableNode);

            const singleTap = new Hammer.Tap({event: 'tap'});
            const doubleTap = new Hammer.Tap({
                event: 'doubletap',
                taps: 2,
                posThreshold: 30,
            });

            this.hammer.add([doubleTap, singleTap]);
            doubleTap.recognizeWith(singleTap);
            singleTap.requireFailure([doubleTap]);

            this.hammer.get('pinch').set({enable: true});
            this.hammer.on('pinchstart', this.handlePinchStart);
            this.hammer.on('pinchmove', this.handlePinchMove);
            this.hammer.on('pinchend', this.handlePinchEnd);

            this.hammer.get('pan').set({direction: Hammer.DIRECTION_ALL});
            this.hammer.on('panstart', this.handlePanStart);
            this.hammer.on('panmove', this.handlePanMove);
            this.hammer.on('panend', this.handlePanEnd);

            this.hammer.on('tap', this.handleTap);
            this.hammer.on('doubletap', this.handleDoubleTap);
        }
    }

    protected removeListeners() {
        if (this.hammer) {
            this.hammer.destroy();
            this.hammer = undefined;
        }
    }

    private handlePinchStart = () => {
        this.disableClick = true;
    };

    private handlePinchMove = (e: HammerInput) => {
        const scaleDelta = e.scale - this.prevScale;

        this.prevScale = e.scale;
        this.changeScale(scaleDelta * this.state.scale, e.center);
    };

    private handlePinchEnd = () => {
        this.prevScale = 1;
    };

    private handlePanStart = (e: HammerInput) => {
        this.beingDrag(e.center);
    };

    private handlePanMove = (e: HammerInput) => {
        this.dragTo(e.center);
    };

    private handlePanEnd = (e: HammerInput) => {
        if (!this.disableClick && e.pointers.length === 1) {
            this.tryCallOnClick(pageToPoint(e.srcEvent as PagePoint), e.center);
        }

        this.endDrag();
        this.disableClick = false;
    };

    private handleTap = (e: HammerInput) => {
        this.tryCallOnClick(pageToPoint(e.srcEvent as PagePoint), e.center);
        this.disableClick = false;
    };

    private handleDoubleTap = (e: HammerInput) => {
        const {scale, minScale} = this.state;

        if (scale >= this.maxScale) {
            this.scaleToCenter(minScale, this.maxScale - minScale);
        } else {
            this.scaleToPoint(this.maxScale, e.center);
        }
    };
}
