import React from 'react';
import {Route, RouteComponentProps, Switch, withRouter} from 'react-router-dom';
import {hot as reactHotLoaderDecorator} from 'react-hot-loader';
import {connect} from 'react-redux';
import {compose} from 'redux';

import {EProjectName} from 'constants/common';
import {ZINDEX_GROUP_LEVEL} from 'constants/layers';
import {URLs} from 'constants/urls';

import {fetchUserPlusInfo} from 'reducers/common/userInfo/thunk';
import {fetchIsSubscribed} from 'reducers/common/subscription/status/thunk';

import appSelector from 'selectors/appSelector';

import {params} from 'utilities/metrika';
import handleMetrikaQuery from 'utilities/metrika/handleMetrikaQuery';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import checkIntlSupport from 'containers/App/utilities/checkIntlSupport';
import checkCssVariablesSupport from 'containers/App/utilities/checkCssVariablesSupport';
import {prefetchData} from 'containers/App/utilities/prefetchData';

import TestAnchors from 'containers/App/components/TestAnchors/TestAnchors';
import MetrikaScript from 'containers/App/components/MetrikaScript/MetrikaScript';
import getProjectPageLoadable from 'containers/App/components/ProjectPageLoadable/ProjectPageLoadable';
import YtpRefererSaver from 'containers/App/components/YtpRefererSaver/YtpRefererSaver';
import BugScriptLoadable from 'containers/App/components/BugScript/BugScriptLoadable';
import FontsSwapper from 'containers/App/components/FontsSwapper/FontsSwapper';

import BaseTravelHelmet from 'components/BaseTravelHelmet/BaseTravelHelmet';
import UxfeedbackLoadable from './components/Uxfeedback/UxfeedbackLoadable';
import PageErrorBoundary from './components/PageErrorBoundary/PageErrorBoundary';
import BottomSheetManager from 'components/BottomSheet/components/BottomSheetManager/BottomSheetManager';

import LayersContext from 'contexts/LayersContext';
import FeaturesContext from 'contexts/FeaturesContext';
import CurrenciesContext from 'contexts/CurrenciesContext';
import DeviceTypeContext from 'contexts/DeviceTypeContext';
import EnvironmentConfigContext from 'contexts/EnvironmentConfigContext';
import {RumUiProvider} from 'contexts/RumUiContext';
import {serverFetchDataDispatcher} from 'contexts/ServerFetchDataContext';

import cx from './App.scss';

if (process.env.DEBUG_LEGO_THEMEKIT) {
    import('./utilities/themeKitDebugStyles');
}

type TAppProps = ReturnType<typeof appSelector> & any & RouteComponentProps;

class App extends React.Component<TAppProps> {
    componentDidMount(): void {
        const {
            dispatch,
            location,
            features,
            userIsAuth,
            userHasPlusInfo,
            subscribeStatusFetched,
        } = this.props;
        const query = getQueryByLocation(location);

        this.trySendHero();
        handleMetrikaQuery(query);
        params({
            features,
            common: {
                userLoggedIn: userIsAuth,
            },
        });

        checkIntlSupport();
        checkCssVariablesSupport();

        if (userIsAuth && !userHasPlusInfo) {
            dispatch(fetchUserPlusInfo());
        }

        if (!subscribeStatusFetched) {
            dispatch(fetchIsSubscribed());
        }
    }

    trySendHero(): void {
        window.Ya?.Rum?.sendHeroElement?.();
    }

    renderProject(project: EProjectName): () => React.ReactNode {
        const Component = getProjectPageLoadable(project);

        return (): React.ReactNode => (
            <PageErrorBoundary project={project}>
                <Component />
            </PageErrorBoundary>
        );
    }

    renderProjects(): React.ReactNode {
        const indexAppPaths = [URLs.index, URLs.hotels];

        return (
            <Switch>
                <Route
                    exact
                    path={indexAppPaths}
                    render={this.renderProject(EProjectName.INDEX)}
                />
                <Route
                    path={URLs.hotels}
                    render={this.renderProject(EProjectName.HOTELS)}
                />
                <Route
                    path={URLs.hotel}
                    render={this.renderProject(EProjectName.HOTEL)}
                />
                <Route render={this.renderProject(EProjectName.NOT_FOUND)} />
            </Switch>
        );
    }

    render(): React.ReactNode {
        const {
            features,
            deviceType,
            currencies,
            environmentConfig,
            needToShowBug,
            isFromXredirect,
        } = this.props;
        const {isMobile} = deviceType;

        return (
            <EnvironmentConfigContext.Provider value={environmentConfig}>
                <DeviceTypeContext.Provider value={deviceType}>
                    <FeaturesContext.Provider value={features}>
                        <CurrenciesContext.Provider value={currencies}>
                            <LayersContext.Provider value={ZINDEX_GROUP_LEVEL}>
                                <BottomSheetManager>
                                    {!isFromXredirect && <BaseTravelHelmet />}
                                    {!__DEV__ && <MetrikaScript />}
                                    {needToShowBug && <BugScriptLoadable />}
                                    <FontsSwapper />
                                    <div
                                        className={cx('App', {
                                            App_mobile: isMobile,
                                        })}
                                    >
                                        <TestAnchors />
                                        <RumUiProvider>
                                            <div>Hello Rooms</div>
                                        </RumUiProvider>
                                    </div>
                                    <YtpRefererSaver />
                                    <UxfeedbackLoadable />
                                </BottomSheetManager>
                            </LayersContext.Provider>
                        </CurrenciesContext.Provider>
                    </FeaturesContext.Provider>
                </DeviceTypeContext.Provider>
            </EnvironmentConfigContext.Provider>
        );
    }
}

export default compose(
    reactHotLoaderDecorator(module),
    withRouter,
    connect(appSelector),
    serverFetchDataDispatcher([prefetchData], {
        checkNested: true,
    }),
)(App);
