import React, {ReactNode} from 'react';

import {EProjectName} from 'constants/common';

import {ECommonGoal} from 'utilities/metrika/types/goals/common';

import {reachGoal} from 'utilities/metrika';
import {logError} from 'utilities/logger/logError';

import * as i18nBlock from 'i18nNew/page-error-boundary';

import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import LayoutError500 from 'components/Layouts/LayoutError500/LayoutError500';

interface IPageErrorBoundaryProps {
    project: EProjectName;
}

interface IPageErrorBoundaryState {
    project?: EProjectName;
    error: boolean;
}

export default class PageErrorBoundary extends React.Component<
    IPageErrorBoundaryProps,
    IPageErrorBoundaryState
> {
    static getDerivedStateFromError(): IPageErrorBoundaryState {
        return {
            error: true,
        };
    }

    static getDerivedStateFromProps(
        props: IPageErrorBoundaryProps,
        state: IPageErrorBoundaryState,
    ): IPageErrorBoundaryState {
        if (props.project !== state.project && state.error) {
            return {
                error: false,
                project: props.project,
            };
        }

        return {
            ...state,
            project: props.project,
        };
    }

    readonly state: IPageErrorBoundaryState = {error: false};

    componentDidCatch(error: Error, info: React.ErrorInfo): void {
        logError(
            {
                message: '[YATRAVEL] Ошибка рендера: ' + error.message,
                block: 'PageErrorBoundary',
                source: info.componentStack,
            },
            error,
        );
        reachGoal(ECommonGoal.APP_RENDER_ERROR);
    }

    render(): ReactNode {
        if (this.state.error) {
            return (
                <LayoutDefault>
                    <LayoutError500
                        title={i18nBlock.title()}
                        subtitle={i18nBlock.subtitle()}
                        action={{
                            type: 'button',
                            title: i18nBlock.actionDashTitle(),
                            handler: (): void =>
                                window.document.location.reload(),
                        }}
                    />
                </LayoutDefault>
            );
        }

        return this.props.children;
    }
}
