import React, {useEffect, useState} from 'react';
import moment from 'moment';
import {noop} from 'lodash';

import {DEFAULT_LANGUAGE_RU} from 'constants/common';

import {IWithClassName} from 'types/withClassName';
import {ESubscriptionCode} from 'types/subscription/ESubscriptionCode';
import {ESubscriptionSource} from 'types/subscription/ESubscriptionSource';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';

import {useAsync} from 'utilities/hooks/useAsync';
import {useToggle} from 'utilities/hooks/useToggle';
import {useIsUserSubscribed} from 'utilities/hooks/subscription/useIsUserSubscribed';

import * as i18nBlock from 'i18nNew/common-mail-subscription';

import Checkbox from 'components/Checkbox/Checkbox';

import {notifierService} from 'serviceProvider';

import cx from './BookingMailSubscription.scss';

interface IBookingMailSubscriptionContainerProps extends IWithClassName {
    email: string;
    subscriptionIsAvailable: boolean;
    travelVerticalName: ESubscriptionVerticalName;
    promoSubscriptionCode: ESubscriptionCode;
    source: ESubscriptionSource;
    checkboxId?: string;
}

export const BookingMailSubscription: React.FC<IBookingMailSubscriptionContainerProps> =
    ({
        className,
        email,
        travelVerticalName,
        promoSubscriptionCode,
        source,
        subscriptionIsAvailable,
        checkboxId,
    }) => {
        const [checked, toggleChecked] = useToggle(true);
        const [disabled, setDisabled] = useState<boolean>(false);
        const isUserSubscribed = useIsUserSubscribed();
        const [, subscribe] = useAsync(
            notifierService.provider()?.subscribe || noop,
        );

        useEffect(() => {
            if (subscriptionIsAvailable) {
                setDisabled(true);

                if (checked) {
                    subscribe({
                        promoSubscriptionCode,
                        email,
                        source,
                        travelVerticalName,
                        timezone: moment.tz.guess(),
                        language: DEFAULT_LANGUAGE_RU,
                        nationalVersion: DEFAULT_LANGUAGE_RU,
                    });
                }
            }
        }, [
            checked,
            subscriptionIsAvailable,
            subscribe,
            email,
            travelVerticalName,
            source,
            setDisabled,
            promoSubscriptionCode,
        ]);

        if (isUserSubscribed) {
            return null;
        }

        return (
            <Checkbox
                className={cx('checkbox', className)}
                id={checkboxId}
                size="s"
                checked={checked}
                disabled={disabled}
                onChange={toggleChecked}
                label={i18nBlock.checkboxDashLabelDash2()}
            />
        );
    };

export default BookingMailSubscription;
