import React, {ReactNode, createContext, useContext} from 'react';

import {ServerDataFetcherBag} from 'server/redux/types';

export type TServerFetchData = (
    props: ServerDataFetcherBag,
) => Promise<unknown> | unknown;

export type TServerFetchDataList = TServerFetchData[];

interface IServerFetchDataProviderProps {
    fetchDataList: TServerFetchDataList;
}

/**
 * Компонент через контекст дает доступ к списку данных, которые нужно загрузить на сервере при первом рендеринге.
 */
export const ServerFetchDataProvider: React.FC<IServerFetchDataProviderProps> =
    props => {
        const {fetchDataList, children} = props;

        return (
            <ServerFetchDataContext.Provider value={fetchDataList}>
                {children}
            </ServerFetchDataContext.Provider>
        );
    };

export const ServerFetchDataContext =
    createContext<TServerFetchDataList | null>(null);

interface IServerFetchDataSetterProps {
    serverFetchList: TServerFetchDataList;
    componentFetchList: TServerFetchDataList;
}

/**
 * Компонет дополняет список с данными, которые надо загрузить на сервере.
 *
 * Рендер на сервере выполнится ровно один раз, поэтому componentFetchList добавится тоже только один раз.
 */
const ServerFetchDataSetter: React.FC<IServerFetchDataSetterProps> = props => {
    const {serverFetchList, componentFetchList} = props;

    serverFetchList.push(...componentFetchList);

    return null;
};

interface IServerFetchDataConsumerProps {
    PageComponent: React.FC<any>;
    checkNested: boolean;
    componentFetchList: TServerFetchDataList;
}

/**
 * Компонент получает из контекста список данных, которые надо загрузить на сервере и дополняет его данными,
 * которые надо загрузить для конкретного компонента из пропсов.
 */
const ServerFetchDataConsumer: React.FC<IServerFetchDataConsumerProps> =
    props => {
        const {
            PageComponent,
            checkNested,
            componentFetchList,
            ...componentProps
        } = props;

        const serverFetchList = useContext(ServerFetchDataContext);

        // Если списка нет, значит это финальный рендеринг html и нам нужно просто отобразить компонент.
        if (!serverFetchList) {
            return <PageComponent {...componentProps} />;
        }

        /*
         * Если для вложенных компонентов не нужно загружать данные при ssr (checkNested === false), то не рендерим PageComponent для оптимизации первого рендеринга на сервере.
         */
        return (
            <>
                <ServerFetchDataSetter
                    serverFetchList={serverFetchList}
                    componentFetchList={componentFetchList}
                />

                {checkNested && <PageComponent {...componentProps} />}
            </>
        );
    };

/**
 * Серверный рендеринг у нас состоит из двух этапов: первый рендеринг части приложения, соответствующий роутингу (определяется по req.url),
 * в ходе которого мы определяем, каких данных нам не хватает для полноценной серверной верстки
 * и второй рендеринг уже с загруженными данными.
 *
 * Собственно данная обертка позволяет указать, какие данные нужно загрузить для конкретного компонента перед финальным рендерингом.
 *
 * Если мы знаем, что абстрактный компонент Comp и его дочерние компонеты не нуждается в подгрузке данных,
 * то мы можем обернуть его `serverFetchDataDispatcher()(Comp)` в целях оптимизации первого рендеринга.
 */
export function serverFetchDataDispatcher(
    componentFetchList: TServerFetchDataList = [],
    {checkNested = false}: {checkNested?: boolean} = {},
) {
    return function <T>(PageComponent: T): T {
        if (__SERVER__) {
            const ServerFetchDataConsumerFactory = (props: any): ReactNode => (
                <ServerFetchDataConsumer
                    PageComponent={PageComponent}
                    checkNested={checkNested}
                    componentFetchList={componentFetchList}
                    {...props}
                />
            );

            // @ts-ignore
            return ServerFetchDataConsumerFactory;
        }

        return PageComponent;
    };
}
