import React, {ComponentType, FunctionComponent, useEffect} from 'react';
import preval from 'preval.macro';

import Box from 'components/Box/Box';
import Input from 'components/Input/Input';
import Heading from 'components/Heading/Heading';

import cx from './IconsShowcase.scss';

// Load icon names on the server side
const iconsDict = preval`
  const fs = require('fs');
  const path = require('path');

  const iconsFolder = path.resolve(__dirname, '..');

  const folders = {
    ['12x12']: path.resolve(iconsFolder, '12'),
    ['16x16']: path.resolve(iconsFolder, '16'),
    ['24x24']: path.resolve(iconsFolder, '24'),
    ['36x36']: path.resolve(iconsFolder, '36'),
  };

  module.exports = Object.entries(folders)
    .reduce((acc, [type, folder]) => ({
        ...acc,
        [type]: fs.readdirSync(folder)
            .filter(icon => /^\.[tj]sx?$/.test(path.extname(icon)))
            .map(icon => {
                const pathObject = path.parse(path.resolve(folder, icon));
                const absolutePath = path.format(pathObject);

                return {
                    name: pathObject.name,
                    path: path.relative(iconsFolder, absolutePath)
                };
            })
  }), {});
`;

interface IIconPath {
    name: string;
    path: string;
}

interface IIconsByName {
    [name: string]: {
        [size: string]: ComponentType;
    };
}

const IconsShowcase: FunctionComponent = () => {
    const [icons, setIcons] = React.useState({} as IIconsByName);
    const [filter, setFilter] = React.useState('');

    useEffect(() => {
        (async function loadIcons(): Promise<void> {
            let iconPaths: IIconPath[] = [];

            iconPaths = iconPaths.concat(iconsDict['12x12']);
            iconPaths = iconPaths.concat(iconsDict['16x16']);
            iconPaths = iconPaths.concat(iconsDict['24x24']);
            iconPaths = iconPaths.concat(iconsDict['36x36']);

            const loadedIcons = await Promise.all(
                iconPaths.map(icon =>
                    import(`../${icon.path}`).then(mod => [
                        icon.name,
                        icon.path,
                        mod.default,
                    ]),
                ),
            );

            const iconsByName: IIconsByName = loadedIcons.reduce(
                (aggr, [name, path, component]) => {
                    aggr[name] = aggr[name] || {};
                    aggr[name][path.slice(0, 2)] = component;

                    return aggr;
                },
                {} as IIconsByName,
            );

            setIcons(iconsByName);
        })();
    }, []);

    const filteredIcons: IIconsByName = React.useMemo(
        () =>
            Object.keys(icons).reduce((filtered, name) => {
                if (name.toLowerCase().includes(filter.toLowerCase())) {
                    filtered[name] = icons[name];
                }

                return filtered;
            }, {} as IIconsByName),
        [icons, filter],
    );

    return (
        <Box inset="4" style={{background: '#f8f8f8'}}>
            <div style={{maxWidth: 640, margin: 'auto'}}>
                <Box below="6">
                    <Heading level={1}>Список иконок</Heading>
                </Box>
                <Box below="4" style={{position: '-webkit-sticky', top: '0'}}>
                    <Input
                        value={filter}
                        placeholder="filter icons"
                        onChange={(evt): void => setFilter(evt.target.value)}
                    />
                </Box>
                <table style={{borderSpacing: 0, width: '100%'}}>
                    <thead>
                        <tr className={cx('row-header')}>
                            <th>
                                <Box inset="2">Имя</Box>
                            </th>

                            <th>
                                <Box inset="2">12</Box>
                            </th>
                            <th>
                                <Box inset="2">16</Box>
                            </th>
                            <th>
                                <Box inset="2">24</Box>
                            </th>
                            <th>
                                <Box inset="2">36</Box>
                            </th>
                        </tr>
                    </thead>
                    <tbody>
                        {Object.entries(filteredIcons).map(
                            ([name, components]) => {
                                const SmallIcon = components['12'];
                                const MediumIcon = components['16'];
                                const LargeIcon = components['24'];
                                const ExtraLargeIcon = components['36'];

                                return (
                                    <tr className={cx('row')} key={name}>
                                        <td>
                                            <Box inset="2">{name}</Box>
                                        </td>
                                        <td>
                                            <Box inset="2">
                                                {SmallIcon && <SmallIcon />}
                                            </Box>
                                        </td>
                                        <td>
                                            <Box inset="2">
                                                {MediumIcon && <MediumIcon />}
                                            </Box>
                                        </td>
                                        <td>
                                            <Box inset="2">
                                                {LargeIcon && <LargeIcon />}
                                            </Box>
                                        </td>
                                        <td>
                                            <Box inset="2">
                                                {ExtraLargeIcon && (
                                                    <ExtraLargeIcon />
                                                )}
                                            </Box>
                                        </td>
                                    </tr>
                                );
                            },
                        )}
                    </tbody>
                </table>
            </div>
        </Box>
    );
};

export default IconsShowcase;
