import React, {ComponentType, FunctionComponent, SVGProps} from 'react';
import preval from 'preval.macro';

import Box from 'components/Box/Box';
import Input from 'components/Input/Input';
import Heading from 'components/Heading/Heading';

import cx from './IllustrationsShowcase.scss';

// Load illustrations names on the server side
// Прочитать все объекты из illustrationsFolder
// Если объект папка, то прочитать из нее иконки и положить их к NEW_ICON_TYPE
// Иначе если объект - это .tsx файл положить его к DEPRECATED_ICON_TYPE
const iconsDict = preval`
  const fs = require('fs');
  const path = require('path');

  const illustrationsFolder = path.resolve(__dirname, '../illustrations');

  function convertIconToNamePath(folder, icon) {
        const pathObject = path.parse(path.resolve(folder, icon));
        const absolutePath = path.format(pathObject);

        return {
            name: pathObject.name,
            path: path.relative(illustrationsFolder, absolutePath)
        };
  }

  function isIconFile(icon) {
        return /^\.tsx?$/.test(path.extname(icon));
  }

  module.exports = fs.readdirSync(illustrationsFolder).reduce((acc, item) => {
        const isDirectory = fs.lstatSync(path.resolve(illustrationsFolder, item))
                              .isDirectory();

        if (isDirectory) {
            const dirPath = path.resolve(illustrationsFolder, item);

            const icons = fs.readdirSync(dirPath)
                .filter(isIconFile)
                .map((icon) => {
                    const iconPath = path.resolve(dirPath, icon);

                    return convertIconToNamePath(dirPath, iconPath);
                });

            acc.push(...icons);
        }

        return acc;
  }, []);
`;

interface IIconsPath {
    name: string;
    path: string;
}

// TODO render by size
interface IIconsByName {
    [name: string]: ComponentType<SVGProps<SVGElement>>;
}

const IconsShowcase: FunctionComponent = () => {
    const [illustrations, setIllustrations] = React.useState(
        {} as IIconsByName,
    );
    const [filter, setFilter] = React.useState('');

    React.useEffect(() => {
        (async function loadIcons(): Promise<void> {
            const iconPaths: IIconsPath[] = iconsDict;

            const loadedIllustrations = await Promise.all(
                iconPaths.map(icon =>
                    import(`../illustrations/${icon.path}`).then(mod => [
                        icon.name,
                        icon.path,
                        mod.default,
                    ]),
                ),
            );

            const illustrationsByName: IIconsByName =
                loadedIllustrations.reduce((aggr, [name, _path, component]) => {
                    aggr[name] = component;

                    return aggr;
                }, {} as IIconsByName);

            setIllustrations(illustrationsByName);
        })();
    }, []);

    const filteredIllustrations: IIconsByName = React.useMemo(
        () =>
            Object.keys(illustrations).reduce((filtered, name) => {
                if (name.toLowerCase().includes(filter.toLowerCase())) {
                    filtered[name] = illustrations[name];
                }

                return filtered;
            }, {} as IIconsByName),
        [illustrations, filter],
    );

    return (
        <Box inset="4" style={{background: '#f8f8f8'}}>
            <div style={{maxWidth: 640, margin: 'auto'}}>
                <Box below="6">
                    <Heading level={1}>Список иллюстраций</Heading>
                </Box>
                <Box below="4" style={{position: '-webkit-sticky', top: '0'}}>
                    <Input
                        value={filter}
                        placeholder="filter icons"
                        onChange={(evt): void => setFilter(evt.target.value)}
                    />
                </Box>
                <Box between="2">
                    {Object.keys(filteredIllustrations).map(name => {
                        const Component = illustrations[name];

                        return (
                            <Box key={name} inset="2" className={cx('item')}>
                                <Heading level={3}>{name}</Heading>
                                <div
                                    className={cx('itemIllustrationContainer')}
                                >
                                    <Component width="100%" height="auto" />
                                </div>
                            </Box>
                        );
                    })}
                </Box>
            </div>
        </Box>
    );
};

export default IconsShowcase;
