import React, {useEffect} from 'react';
import {Route, Switch} from 'react-router-dom';
import {useDispatch} from 'react-redux';

import {URLs} from 'constants/urls';

import {ELoadableReducer} from 'types/common/ELoadableReducer';

import {
    resetOrderInfoAction,
    stopOrderInfoPollingAction,
    stopCreateOrderActions,
    stopOrderPaymentActions,
} from 'reducers/depreacted/hotels/bookAndPayPage/orderInfo/actions';
import bookAndPayPageReducer from 'reducers/depreacted/hotels/bookAndPayPage/bookAndPayReducer';

import getOfferInfoByToken from 'sagas/depreacted/hotels/bookAndPay/getOfferInfoByToken';
import createOrder from 'sagas/depreacted/hotels/bookAndPay/createOrder';
import estimateDiscount from 'sagas/depreacted/hotels/bookAndPay/estimateDiscount';
import pollingOrder from 'sagas/depreacted/hotels/bookAndPay/pollingOrder';
import startPaymentOrder from 'sagas/depreacted/hotels/bookAndPay/startPaymentOrder';
import cancelOrder from 'sagas/depreacted/hotels/bookAndPay/cancelOrder';
import removeOrder from 'sagas/depreacted/hotels/bookAndPay/removeOrder';

import {loadable} from 'utilities/pageLoadable';

import withSaga from 'containers/withSaga/withSaga';
import withReducers from 'containers/withReducers/withReducers';

const BookPage = loadable(
    () => import('projects/depreacted/hotels/pages/BookPage/BookPageContainer'),
);

const PaymentPage = loadable(
    () => import('projects/depreacted/hotels/pages/PaymentPage/PaymentPageContainer'),
);

const FinishPaymentPage = loadable(
    () => import('components/Layouts/PaymentFinish/PaymentFinish'),
);

const HappyPage = loadable(
    () => import('projects/depreacted/hotels/pages/HappyPage/HappyPage'),
);

const BookErrorPage = loadable(
    () => import('projects/depreacted/hotels/pages/BookErrorPage/BookErrorPage'),
);

const CancelledOrderPage = loadable(
    () => import('projects/depreacted/hotels/pages/CancelledOrderPage/CancelledOrderPage'),
);

const AuthorizeVoucherPage = loadable(
    () => import('containers/AuthTicket/AuthTicket'),
);

const HotelsStartDeferredPaymentPage = loadable(
    () =>
        import(
            'projects/depreacted/hotels/pages/HotelsStartDeferredPaymentPage/HotelsStartDeferredPaymentPageContainer'
        ),
);

const NotFoundApp = loadable(
    () => import('components/NotFound/NotFoundPage/NotFoundPage'),
);

interface IHotelsBookAppProps {}

const HotelsBookApp: React.FC<IHotelsBookAppProps> = () => {
    const dispatch = useDispatch();

    useEffect(() => {
        return (): void => {
            dispatch(stopCreateOrderActions());
            dispatch(stopOrderPaymentActions());
            dispatch(stopOrderInfoPollingAction());
            dispatch(resetOrderInfoAction());
        };
    }, [dispatch]);

    return (
        <Switch>
            <Route exact path={URLs.hotelsBook} component={BookPage} />
            <Route path={URLs.hotelsBookPayment} component={PaymentPage} />
            <Route
                path={URLs.hotelsBookFinishPayment}
                component={FinishPaymentPage}
            />
            <Route path={URLs.hotelsBookSuccess} component={HappyPage} />
            <Route path={URLs.hotelsBookError} component={BookErrorPage} />
            <Route
                path={URLs.hotelsCancelledOrder}
                component={CancelledOrderPage}
            />
            <Route
                path={URLs.hotelsAuthorizeVoucher}
                component={AuthorizeVoucherPage}
            />
            <Route
                path={URLs.hotelsBookStartDeferredPayment}
                component={HotelsStartDeferredPaymentPage}
            />
            <Route component={NotFoundApp} />
        </Switch>
    );
};

export default React.memo(
    withReducers([
        [ELoadableReducer.HOTELS_BOOK_AND_PAY, bookAndPayPageReducer],
    ])(
        withSaga([
            getOfferInfoByToken,
            createOrder,
            estimateDiscount,
            pollingOrder,
            startPaymentOrder,
            cancelOrder,
            removeOrder,
        ])(HotelsBookApp),
    ),
);
