import React, {PureComponent, ReactNode, ChangeEvent} from 'react';

import {
    IBookDeferredPaymentSchedule,
    IBookSearchParams,
} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IWithClassName} from 'types/withClassName';
import {ICancellationInfo} from 'types/hotels/offer/IHotelOfferCancellationInfo';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {prepareDeferredPaymentDateTime} from 'projects/depreacted/hotels/utilities/prepareDeferredPaymentDate/prepareDeferredPaymentDate';
import {HUMAN_DATETIME_WITHOUT_YEAR} from 'utilities/dateUtils/formats';
import {CHAR_NBSP} from 'utilities/strings/charCodes';
import {
    IWithQaAttributes,
    prepareQaAttributes,
    prepareCustomQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-BookDeferredPayment';

import Radiobox from 'components/Radiobox/Radiobox';
import Price from 'components/Price/Price';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import LinkButton from 'components/LinkButton/LinkButton';
import ModalOrBottomSheet from 'components/ModalOrBottomSheet/ModalOrBottomSheet';
import DeferredPaymentScheduleInfo from './components/DeferredPaymentScheduleInfo/DeferredPaymentScheduleInfo';

import cx from './BookDeferredPaymentSelect.scss';

interface IBookDeferredPaymentSelectProps
    extends IWithClassName,
        IWithQaAttributes {
    deferredPaymentSchedule: IBookDeferredPaymentSchedule;
    deviceType: IDevice;
    useDeferredPayments?: boolean;
    cancellationInfo: ICancellationInfo;
    searchParams: IBookSearchParams;
    isDisabled: boolean;
    onChangeDeferredPayment?: (useDeferredPayments: boolean) => void;
}

interface IBookDeferredPaymentSelectState {
    isVisiblePaymentScheduleInfo: boolean;
}

enum EPaymentSelectType {
    TOTAL = 'total',
    DEFERRED = 'deferred',
}

const INITIAL_USE_DEFERRED_PAYMENTS_VALUE = true;

class BookDeferredPaymentSelect extends PureComponent<
    IBookDeferredPaymentSelectProps,
    IBookDeferredPaymentSelectState
> {
    state = {
        isVisiblePaymentScheduleInfo: false,
    };

    componentDidMount(): void {
        // при работе внутри формы не всегда подхватывается, если не в конце колл-стэка
        setTimeout(() => {
            this.setInitialPaymentOption();
        }, 0);
    }

    /* Helpers */

    private setInitialPaymentOption(): void {
        const {onChangeDeferredPayment} = this.props;

        if (onChangeDeferredPayment) {
            onChangeDeferredPayment(INITIAL_USE_DEFERRED_PAYMENTS_VALUE);
        }
    }

    private getPaymentLabel({
        zeroFirstPayment,
        percentage,
    }: {
        zeroFirstPayment: boolean;
        percentage: number | undefined;
    }): string {
        if (zeroFirstPayment)
            return i18nBlock.initialPaymentLabelNoPrepayment();

        if (percentage) {
            return i18nBlock.initialPaymentLabel({
                percentNode: percentage,
            });
        }

        return i18nBlock.initialPaymentLabelWithoutPercent();
    }

    /* Handlers */

    private handlePaymentScheduleInfoButtonClick = (
        e: React.MouseEvent,
    ): void => {
        if (e) {
            e.stopPropagation();
            e.preventDefault();
        }

        this.setState({isVisiblePaymentScheduleInfo: true});
    };

    private handleClosePaymentScheduleInfo = (): void => {
        this.setState({isVisiblePaymentScheduleInfo: false});
    };

    private handleChangePaymentOption = (
        e: ChangeEvent<HTMLInputElement>,
    ): void => {
        const {onChangeDeferredPayment} = this.props;
        const paymentValue = e?.target?.value;
        const useDeferredPayments =
            paymentValue === EPaymentSelectType.DEFERRED;

        if (onChangeDeferredPayment) {
            onChangeDeferredPayment(useDeferredPayments);
        }
    };

    /* Render */

    private renderPaymentScheduleInfo(): ReactNode {
        const {isVisiblePaymentScheduleInfo} = this.state;
        const {
            deviceType,
            cancellationInfo,
            deferredPaymentSchedule,
            searchParams,
        } = this.props;

        return (
            <ModalOrBottomSheet
                isVisible={isVisiblePaymentScheduleInfo}
                onClose={this.handleClosePaymentScheduleInfo}
                modalContentProps={{className: cx('deferredPaymentModal')}}
            >
                <DeferredPaymentScheduleInfo
                    deviceType={deviceType}
                    cancellationInfo={cancellationInfo}
                    deferredPaymentSchedule={deferredPaymentSchedule}
                    searchParams={searchParams}
                />
            </ModalOrBottomSheet>
        );
    }

    private renderDeferredPaymentOption(): ReactNode {
        const {
            useDeferredPayments,
            deferredPaymentSchedule: {
                initialPayment,
                deferredPayments,
                zeroFirstPayment,
            },
        } = this.props;
        const [deferredPayment] = deferredPayments;

        return (
            <Flex
                flexDirection="column"
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'deferredPaymentCheckbox',
                })}
                tagProps={{
                    ...prepareCustomQaAttributes(
                        'data-active',
                        useDeferredPayments ? 'true' : 'false',
                    ),
                }}
            >
                <Flex
                    className={cx('paymentLabel')}
                    justifyContent="space-between"
                    alignItems="baseline"
                >
                    <Text
                        className={cx('label')}
                        size="s"
                        color="primary"
                        weight="medium"
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'postpayRadioboxLabel',
                        })}
                    >
                        {this.getPaymentLabel({
                            zeroFirstPayment,
                            percentage: initialPayment?.percentage,
                        })}
                    </Text>
                    {!zeroFirstPayment && (
                        <Text size="s" color="primary">
                            <Price
                                value={initialPayment.amount.value}
                                currency={initialPayment.amount.currency}
                                {...prepareQaAttributes({
                                    parent: this.props,
                                    current: 'prepayPrice',
                                })}
                            />
                        </Text>
                    )}
                </Flex>
                <Text size="s" color="primary">
                    {zeroFirstPayment
                        ? insertJSXIntoKey(
                              i18nBlock.deferredPaymentInfoNoPrepayment,
                          )({
                              dateNode: (
                                  <span
                                      className={cx('dateLabel')}
                                      {...prepareQaAttributes({
                                          parent: this.props,
                                          current: 'paymentEndsAt',
                                      })}
                                  >
                                      {prepareDeferredPaymentDateTime(
                                          deferredPayment.paymentEndsAt,
                                          HUMAN_DATETIME_WITHOUT_YEAR,
                                      )}
                                  </span>
                              ),
                          })
                        : insertJSXIntoKey(i18nBlock.deferredPaymentInfo)({
                              priceNode: (
                                  <Price
                                      value={deferredPayment.amount.value}
                                      currency={deferredPayment.amount.currency}
                                      {...prepareQaAttributes({
                                          parent: this.props,
                                          current: 'postpayPrice',
                                      })}
                                  />
                              ),
                              dateNode: (
                                  <span
                                      className={cx('dateLabel')}
                                      {...prepareQaAttributes({
                                          parent: this.props,
                                          current: 'paymentEndsAt',
                                      })}
                                  >
                                      {prepareDeferredPaymentDateTime(
                                          deferredPayment.paymentEndsAt,
                                          HUMAN_DATETIME_WITHOUT_YEAR,
                                      )}
                                  </span>
                              ),
                          })}
                    {CHAR_NBSP}
                    <LinkButton
                        onClick={this.handlePaymentScheduleInfoButtonClick}
                    >
                        {i18nBlock.moreInfoLabel()}
                    </LinkButton>
                </Text>
            </Flex>
        );
    }

    private renderTotalPaymentOption(): ReactNode {
        return (
            <Flex
                justifyContent="space-between"
                alignItems="baseline"
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'fullPaymentCheckbox',
                })}
            >
                <Text size="s" color="primary" weight="medium">
                    {i18nBlock.nowPaymentLabel()}
                </Text>
            </Flex>
        );
    }

    render(): ReactNode {
        const {
            className,
            deviceType: {isDesktop},
            useDeferredPayments,
            isDisabled,
        } = this.props;

        return (
            <div className={cx(className)}>
                <Radiobox
                    onChange={this.handleChangePaymentOption}
                    size={isDesktop ? 's' : 'm'}
                    value={
                        useDeferredPayments
                            ? EPaymentSelectType.DEFERRED
                            : EPaymentSelectType.TOTAL
                    }
                    options={[
                        {
                            value: EPaymentSelectType.DEFERRED,
                            label: this.renderDeferredPaymentOption(),
                        },
                        {
                            value: EPaymentSelectType.TOTAL,
                            label: this.renderTotalPaymentOption(),
                        },
                    ]}
                    device={isDesktop ? 'desktop' : 'touch'}
                    disabled={isDisabled}
                />
                {this.renderPaymentScheduleInfo()}
            </div>
        );
    }
}

export default BookDeferredPaymentSelect;
