import React from 'react';

import {
    ICancellationInfo,
    ERefundType,
} from 'types/hotels/offer/IHotelOfferCancellationInfo';
import {IBookDeferredPaymentSchedule} from 'server/api/HotelsBookAPI/types/IBookOffer';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {
    HUMAN_WITH_YEAR,
    TIME,
    ROBOT_DATETIME,
} from 'utilities/dateUtils/formats';
import {prepareDeferredPaymentDateTime} from 'projects/depreacted/hotels/utilities/prepareDeferredPaymentDate/prepareDeferredPaymentDate';

import * as i18nBlock from 'i18n/hotels-DeferredPaymentScheduleInfo';

function getIfForgetToPayTheRestAnswer({
    deferredPaymentEndsAtDate,
    deferredPaymentEndsAtTime,
    penaltyIfUnpaidPrice,
    prepaymentPrice,
    refundType,
    zeroFirstPayment,
}: {
    deferredPaymentEndsAtDate: string;
    deferredPaymentEndsAtTime: string;
    penaltyIfUnpaidPrice: React.ReactElement;
    prepaymentPrice: React.ReactElement;
    refundType: ERefundType;
    zeroFirstPayment?: boolean;
}): React.ReactElement | string {
    if (zeroFirstPayment)
        return insertJSXIntoKey(
            i18nBlock.ifForgetToPayTheRestAnswerNoPrepayment,
        )({
            date: deferredPaymentEndsAtDate,
            time: deferredPaymentEndsAtTime,
        });

    switch (refundType) {
        case ERefundType.REFUNDABLE_WITH_PENALTY:
            return insertJSXIntoKey(
                i18nBlock.ifForgetToPayTheRestAnswerRefundableWithPenalty,
            )({
                date: deferredPaymentEndsAtDate,
                time: deferredPaymentEndsAtTime,
                amountFine: penaltyIfUnpaidPrice,
            });
        case ERefundType.NON_REFUNDABLE:
            return i18nBlock.ifForgetToPayTheRestAnswerNonRefundable({
                date: deferredPaymentEndsAtDate,
                time: deferredPaymentEndsAtTime,
            });
        case ERefundType.FULLY_REFUNDABLE:
            return insertJSXIntoKey(
                i18nBlock.ifForgetToPayTheRestAnswerFreeCancellation,
            )({
                date: deferredPaymentEndsAtDate,
                time: deferredPaymentEndsAtTime,
                value: prepaymentPrice,
            });
        default:
            return '';
    }
}

function howCancelReservationForFreeAnswer({
    refundType,
    isEndOfFreeCancellationEndsBeforePaymentEndsAt,
    endOfFreeCancellationDate,
    endOfFreeCancellationTime,
    prepaymentPrice,
    deferredPaymentEndsAtDate,
    deferredPaymentEndsAtTime,
    zeroFirstPayment,
}: {
    refundType: ERefundType;
    isEndOfFreeCancellationEndsBeforePaymentEndsAt: boolean;
    endOfFreeCancellationDate: string;
    endOfFreeCancellationTime: string;
    prepaymentPrice: React.ReactElement;
    deferredPaymentEndsAtDate: string;
    deferredPaymentEndsAtTime: string;
    zeroFirstPayment?: boolean;
}): React.ReactElement | string {
    if (zeroFirstPayment)
        return insertJSXIntoKey(
            i18nBlock.howCancelReservationForFreeAnswerNoPrepayment,
        )({
            date: deferredPaymentEndsAtDate,
            time: deferredPaymentEndsAtTime,
        });

    if (
        refundType === ERefundType.REFUNDABLE_WITH_PENALTY &&
        isEndOfFreeCancellationEndsBeforePaymentEndsAt
    ) {
        return insertJSXIntoKey(
            i18nBlock.howCancelReservationForFreeAnswerRefundableWithPenalty,
        )({
            date: endOfFreeCancellationDate,
            time: endOfFreeCancellationTime,
            value: prepaymentPrice,
        });
    }

    if (refundType === ERefundType.NON_REFUNDABLE) {
        return i18nBlock.howCancelReservationForFreeAnswerNonRefundable();
    }

    if (
        refundType === ERefundType.FULLY_REFUNDABLE ||
        refundType === ERefundType.REFUNDABLE_WITH_PENALTY
    ) {
        return insertJSXIntoKey(
            i18nBlock.howCancelReservationForFreeAnswerFreeCancellation,
        )({
            date: deferredPaymentEndsAtDate,
            time: deferredPaymentEndsAtTime,
            value: prepaymentPrice,
        });
    }

    return '';
}

interface IGetDeferredPaymentQuestionsAndAnswersProps {
    deferredPaymentSchedule: IBookDeferredPaymentSchedule;
    cancellationInfo: ICancellationInfo;
    prepaymentPrice: React.ReactElement;
    penaltyIfUnpaidPrice: React.ReactElement;
}

enum EDeferredPaymentAnswersType {
    HOW_TO_BOOK = 'howToBook',
    HOW_PAY_THE_REST = 'howPayTheRest',
    IF_FORGET_TO_PAY_THE_REST = 'ifForgetToPayTheRest',
    HOW_CANCEL_RESERVATION_FOR_FREE = 'howCancelReservationForFree',
}

interface IDeferredPaymentQuestionAndAnswer {
    type: EDeferredPaymentAnswersType;
    questionText: string;
    answerText: React.ReactElement | string;
}

export default function getDeferredPaymentQuestionsAndAnswers({
    deferredPaymentSchedule,
    cancellationInfo,
    prepaymentPrice,
    penaltyIfUnpaidPrice,
}: IGetDeferredPaymentQuestionsAndAnswersProps): IDeferredPaymentQuestionAndAnswer[] {
    const {
        deferredPayments: [deferredPayment],
        zeroFirstPayment,
    } = deferredPaymentSchedule;

    const {paymentEndsAt} = deferredPayment;

    const deferredPaymentEndsAtDate = prepareDeferredPaymentDateTime(
        paymentEndsAt,
        HUMAN_WITH_YEAR,
    );
    const deferredPaymentEndsAtTime = prepareDeferredPaymentDateTime(
        paymentEndsAt,
        TIME,
    );

    const {refundType, refundRules} = cancellationInfo;

    const endOfFreeCancellation: string | null = refundRules?.length
        ? refundRules[0].startsAt
        : null;
    const endOfFreeCancellationDate = endOfFreeCancellation
        ? prepareDeferredPaymentDateTime(endOfFreeCancellation, HUMAN_WITH_YEAR)
        : '';
    const endOfFreeCancellationTime = endOfFreeCancellation
        ? prepareDeferredPaymentDateTime(endOfFreeCancellation, TIME)
        : '';
    const isEndOfFreeCancellationEndsBeforePaymentEndsAt = Boolean(
        refundType === ERefundType.REFUNDABLE_WITH_PENALTY &&
            endOfFreeCancellation &&
            prepareDeferredPaymentDateTime(
                endOfFreeCancellation,
                ROBOT_DATETIME,
            ) < prepareDeferredPaymentDateTime(paymentEndsAt, ROBOT_DATETIME),
    );

    return zeroFirstPayment
        ? [
              {
                  type: EDeferredPaymentAnswersType.HOW_TO_BOOK,
                  questionText: i18nBlock.howToBookNoPrepayment(),
                  answerText: insertJSXIntoKey(
                      i18nBlock.howToBookAnswerNoPrepayment,
                  )({
                      value: prepaymentPrice,
                  }),
              },
              {
                  type: EDeferredPaymentAnswersType.HOW_PAY_THE_REST,
                  questionText: i18nBlock.howPayTheRestNoPrepayment(),
                  answerText: insertJSXIntoKey(
                      i18nBlock.howPayTheRestAnswerNoPrepayment,
                  )({
                      date: deferredPaymentEndsAtDate,
                      time: deferredPaymentEndsAtTime,
                  }),
              },
              {
                  type: EDeferredPaymentAnswersType.IF_FORGET_TO_PAY_THE_REST,
                  questionText: i18nBlock.ifForgetToPayTheRestNoPrepayment(),
                  answerText: getIfForgetToPayTheRestAnswer({
                      deferredPaymentEndsAtDate,
                      deferredPaymentEndsAtTime,
                      penaltyIfUnpaidPrice,
                      prepaymentPrice,
                      refundType,
                      zeroFirstPayment,
                  }),
              },
              {
                  type: EDeferredPaymentAnswersType.HOW_CANCEL_RESERVATION_FOR_FREE,
                  questionText: i18nBlock.howCancelReservationForFree(),
                  answerText: howCancelReservationForFreeAnswer({
                      refundType,
                      isEndOfFreeCancellationEndsBeforePaymentEndsAt,
                      endOfFreeCancellationDate,
                      endOfFreeCancellationTime,
                      prepaymentPrice,
                      deferredPaymentEndsAtDate,
                      deferredPaymentEndsAtTime,
                      zeroFirstPayment,
                  }),
              },
          ]
        : [
              {
                  type: EDeferredPaymentAnswersType.HOW_TO_BOOK,
                  questionText: i18nBlock.howToBook(),
                  answerText: insertJSXIntoKey(i18nBlock.howToBookAnswer)({
                      value: prepaymentPrice,
                  }),
              },
              {
                  type: EDeferredPaymentAnswersType.HOW_PAY_THE_REST,
                  questionText: i18nBlock.howPayTheRest(),
                  answerText: insertJSXIntoKey(i18nBlock.howPayTheRestAnswer)({
                      date: deferredPaymentEndsAtDate,
                      time: deferredPaymentEndsAtTime,
                  }),
              },
              {
                  type: EDeferredPaymentAnswersType.IF_FORGET_TO_PAY_THE_REST,
                  questionText: i18nBlock.ifForgetToPayTheRest(),
                  answerText: getIfForgetToPayTheRestAnswer({
                      deferredPaymentEndsAtDate,
                      deferredPaymentEndsAtTime,
                      penaltyIfUnpaidPrice,
                      prepaymentPrice,
                      refundType,
                  }),
              },
              {
                  type: EDeferredPaymentAnswersType.HOW_CANCEL_RESERVATION_FOR_FREE,
                  questionText: i18nBlock.howCancelReservationForFree(),
                  answerText: howCancelReservationForFreeAnswer({
                      refundType,
                      isEndOfFreeCancellationEndsBeforePaymentEndsAt,
                      endOfFreeCancellationDate,
                      endOfFreeCancellationTime,
                      prepaymentPrice,
                      deferredPaymentEndsAtDate,
                      deferredPaymentEndsAtTime,
                  }),
              },
          ];
}
