import {
    IBookHotelInfo,
    IBookSearchParams,
} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IWithDeviceType} from 'types/withDeviceType';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {hotelsURLs} from 'projects/depreacted/hotels/utilities/urls';

import * as i18nBlock from 'i18n/hotels-BookPayPage';
import * as i18nSupportBlock from 'i18n/common-support';

import {useSupportPhone} from 'hooks/useSupportPhone';

import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import PhoneWithTitle from 'components/PhoneWithTitle/PhoneWithTitle';

import cx from './BookHeader.scss';

const stepTextByType: Record<string, string> = {
    book: i18nBlock.shortBookStep(),
    pay: i18nBlock.payStep(),
    confirm: i18nBlock.confirmStep(),
};

export const BOOK_STEPS = ['book', 'pay'];
export const BOOK_STEPS_NO_PREPAYMENT = ['book'];

interface IBookHeaderProps extends IWithQaAttributes, IWithDeviceType {
    hotelInfo: IBookHotelInfo;
    isNoPrepayment?: boolean;
    searchParams: IBookSearchParams;
    currentStep: string;
}

const BookHeader: React.FC<IBookHeaderProps> = props => {
    const {
        hotelInfo,
        searchParams,
        currentStep = BOOK_STEPS[0],
        deviceType: {isMobile},
        isNoPrepayment,
    } = props;
    const supportPhone = useSupportPhone();

    return (
        <BookingLayout.Breadcrumbs
            additional={
                !isMobile && (
                    <BookingLayout.Breadcrumbs.SupportItem>
                        <PhoneWithTitle
                            title={i18nSupportBlock.title()}
                            phone={supportPhone}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'support-phone',
                            })}
                        />
                    </BookingLayout.Breadcrumbs.SupportItem>
                )
            }
        >
            {hotelInfo.breadcrumbs &&
                hotelInfo.breadcrumbs.geoRegions &&
                hotelInfo.breadcrumbs.geoRegions.map(({geoId, linguistics}) => (
                    <BookingLayout.Breadcrumbs.Item
                        className={cx('item')}
                        key={geoId}
                        to={hotelsURLs.getHotelsSearchUrlByParams(geoId, {
                            ...searchParams,
                            checkinDate: searchParams.startDate,
                            checkoutDate: searchParams.endDate,
                        })}
                    >
                        {linguistics.nominativeCase}
                    </BookingLayout.Breadcrumbs.Item>
                ))}
            <BookingLayout.Breadcrumbs.Item
                className={cx('item')}
                key={hotelInfo.permalink}
                to={hotelsURLs.getHotelUrl({
                    ...searchParams,
                    hotelSlug: hotelInfo.hotelSlug,
                    checkinDate: searchParams.startDate,
                    checkoutDate: searchParams.endDate,
                })}
            >
                {hotelInfo.name}
            </BookingLayout.Breadcrumbs.Item>
            {(isNoPrepayment ? BOOK_STEPS_NO_PREPAYMENT : BOOK_STEPS).map(
                (step, i) => (
                    <BookingLayout.Breadcrumbs.Item
                        className={cx('item')}
                        key={i}
                        active={currentStep === step}
                        disabled={step === 'pay'}
                    >
                        {stepTextByType[step]}
                    </BookingLayout.Breadcrumbs.Item>
                ),
            )}
        </BookingLayout.Breadcrumbs>
    );
};

export default BookHeader;
