import {PureComponent, ReactNode} from 'react';
import ReactTruncate from 'react-truncate';

import {IWithClassName} from 'types/withClassName';
import {TBookOfferAmenities} from 'server/api/HotelsBookAPI/types/IBookOffer';
import IAmenity from 'server/api/GenericOrderApi/types/common/service/IHotelServiceInfo/IHotelOfferInfo/IAmenity';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {
    CHAR_ENSP,
    CHAR_LIST_MARKER_THIN,
    CHAR_NBSP,
} from 'utilities/strings/charCodes';

import * as i18nBlock from 'i18n/hotels-BookHotelAmenities';

import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import HotelFeaturesIcon from 'icons/16/HotelFeatures';

import cx from './BookHotelAmenities.scss';

interface IBookHotelAmenitiesProps extends IWithClassName, IWithQaAttributes {
    amenities: TBookOfferAmenities;
}

interface IBookHotelAmenitiesState {
    canRenderAllLines: boolean;
}

const COLLAPSED_LINES_COUNT = 3;

class BookHotelAmenities extends PureComponent<
    IBookHotelAmenitiesProps,
    IBookHotelAmenitiesState
> {
    static defaultProps = {
        className: '',
        amenities: {},
    };

    state = {
        canRenderAllLines: false,
    };

    /**
     * костыль под то, что по какой-то причине хромиум рендерит неразрывный пробел
     * очень узким, а остальные браузеры как полноценный пробел
     */
    get separatorSpaceCount(): number {
        const isChromium = Boolean(window.chrome);

        return isChromium ? 3 : 1;
    }

    private getAmenitiesList(): IAmenity[] {
        const {amenities} = this.props;

        return Object.values(amenities);
    }

    private handleTruncateMoreButtonClick = (): void => {
        this.setState(prevState => ({
            canRenderAllLines: !prevState.canRenderAllLines,
        }));
    };

    private renderTruncateEllipsis = (): ReactNode => {
        return (
            <span className={cx('truncateEllipsis')}>
                <span className={cx('truncateEllipsisDots')}>...</span>
            </span>
        );
    };

    private renderItem = (
        {id, name}: IAmenity,
        index: number,
        array: IAmenity[],
    ): ReactNode => {
        const before = CHAR_NBSP.repeat(this.separatorSpaceCount);
        const middleDotNode = (
            <span>
                {before}
                {CHAR_LIST_MARKER_THIN}
                {CHAR_ENSP}
            </span>
        );

        return (
            <span key={id}>
                {name}
                {index === array.length - 1 ? null : middleDotNode}
            </span>
        );
    };

    private renderList(): ReactNode {
        const {canRenderAllLines} = this.state;
        const amenitiesList = this.getAmenitiesList();

        return (
            <>
                <ReactTruncate
                    lines={canRenderAllLines ? false : COLLAPSED_LINES_COUNT}
                    ellipsis={this.renderTruncateEllipsis()}
                >
                    {amenitiesList.map(this.renderItem)}
                </ReactTruncate>
                <br />
                <button
                    className={cx('truncateEllipsisMore', 'pt')}
                    type="button"
                    onClick={this.handleTruncateMoreButtonClick}
                >
                    {canRenderAllLines
                        ? i18nBlock.hideButtonText()
                        : i18nBlock.moreButtonText()}
                </button>
            </>
        );
    }

    render(): ReactNode {
        const {className} = this.props;
        const amenitiesList = this.getAmenitiesList();
        const canRenderHotelAmenities = amenitiesList.length > 0;

        return (
            canRenderHotelAmenities && (
                <section
                    className={className}
                    {...prepareQaAttributes(this.props)}
                >
                    <TextWithIcon
                        iconLeft={HotelFeaturesIcon}
                        text={this.renderList()}
                        textClassName={cx('truncateEllipsisContent')}
                    />
                </section>
            )
        );
    }
}

export default BookHotelAmenities;
