import {Component, ReactNode} from 'react';
import _noop from 'lodash/noop';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {
    IBookSearchParams,
    IBookHotelInfo,
    IBookOfferBedsGroup,
    IBookOfferCancellationInfo,
    IBookOfferMeal,
    IBookOfferRoom,
    IBookPartnerHotelInfo,
} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IBookUserSelect} from 'types/hotels/book/ICreateOrder';
import {EHotelImageAltayStandardSize} from 'types/hotels/common/EHotelImageAltayStandardSize';

import {deviceMods} from 'utilities/stylesUtils';
import {getHotelNameWithStars} from 'projects/depreacted/hotels/utilities/getHotelNameWithStars/getHotelNameWithStars';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {parseImageUrlTemplate} from 'projects/depreacted/hotels/utilities/prepareAndParseImages/prepareAndParseImages';

import * as i18nBlock from 'i18n/hotels-BookPayPage';

import Image from 'components/TravelImage/TravelImage';
import LinkButton from 'components/LinkButton/LinkButton';
import BookSettlementInfo from 'projects/depreacted/hotels/components/BookSettlementInfo/BookSettlementInfo';
import BookSelectBedsGroups from 'projects/depreacted/hotels/components/BookSelectBedsGroups/BookSelectBedsGroups';
import BookHotelAmenities from 'projects/depreacted/hotels/components/BookHotelAmenities/BookHotelAmenities';
import OfferRefundableRules from 'projects/depreacted/hotels/components/OfferRefundableRules/OfferRefundableRulesWithTrigger';
import BookMealInfo from 'projects/depreacted/hotels/components/BookMealInfo/BookMealInfo';
import BookPartnerHotelInfo from 'projects/depreacted/hotels/components/BookPartnerHotelInfo/BookPartnerHotelInfo';
import HotelAddressAndRating from 'projects/depreacted/hotels/components/HotelAddressAndRating/HotelAddressAndRating';
import BookSearchParams from 'projects/depreacted/hotels/components/BookSearchParams/BookSearchParams';
import BookRoomDescription from 'projects/depreacted/hotels/components/BookRoomDescription/BookRoomDescription';
import {TwoColumnLayout} from 'components/Layouts/TwoColumnLayout/TwoColumnLayout';

import cx from './BookHotelInfo.scss';

/* Constants */
const HOTEL_NAME_SYMBOLS_COUNT_FOR_SIZE_L = 80;

export interface IBookHotelInfoProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    searchParams: IBookSearchParams;
    hotelInfo: IBookHotelInfo;
    roomInfo: IBookOfferRoom;
    cancellationInfo: IBookOfferCancellationInfo;
    mealInfo: IBookOfferMeal | undefined;
    partnerHotelInfo: IBookPartnerHotelInfo;
    bookUserSelect: IBookUserSelect;

    bedsGroups: IBookOfferBedsGroup[];
    setBedsGroupId(): void;

    isExpiredOffer: boolean;
    preventOpacityOverlay: boolean;
    footerNode: ReactNode;
}

class BookHotelInfo extends Component<IBookHotelInfoProps> {
    static defaultProps = {
        className: '',
        footerNode: null,
        hotelInfo: {},
        searchParams: {},
        roomInfo: {},
        partnerHotelInfo: {},
        cancellationInfo: {},
        bedsGroups: [],
        bookUserSelect: {},
        setBedsGroupId: _noop,
        preventOpacityOverlay: false,
        isExpiredOffer: false,
    };

    state = {
        canRenderPartnerInfoModal: false,
    };

    /* Handlers */

    private handleClickPartnerModalTrigger = (): void => {
        this.showPartnerInfoModal();
    };

    private showPartnerInfoModal = (): void => {
        this.setState({canRenderPartnerInfoModal: true});
    };

    private hidePartnerInfoModal = (): void => {
        this.setState({canRenderPartnerInfoModal: false});
    };

    private renderHotelBaseInfo(): ReactNode {
        const {hotelInfo, deviceType} = this.props;

        const {name, stars} = hotelInfo;

        const preparedHotelName = getHotelNameWithStars({
            name,
            stars,
        });

        return (
            <div>
                {deviceType.isMobile && this.renderHotelImage()}
                {Boolean(preparedHotelName) && (
                    <div
                        className={cx('hotelName', {
                            hotelName_size_smaller:
                                preparedHotelName.length >
                                HOTEL_NAME_SYMBOLS_COUNT_FOR_SIZE_L,
                        })}
                    >
                        <span
                            className={cx('hotelNameLink')}
                            onClick={this.handleClickPartnerModalTrigger}
                            {...prepareQaAttributes({
                                current: 'hotelNameLink',
                            })}
                        >
                            {preparedHotelName}
                        </span>
                    </div>
                )}
                <HotelAddressAndRating
                    className={cx('hotelAddressRating')}
                    hotelInfo={hotelInfo}
                    onAddressClick={this.handleClickPartnerModalTrigger}
                    addressType="link"
                    multiLine
                    deviceType={deviceType}
                    {...prepareQaAttributes({
                        current: 'addressAndRating',
                    })}
                />
            </div>
        );
    }

    private renderSearchParams(): ReactNode {
        const {hotelInfo, searchParams} = this.props;

        return (
            <BookSearchParams
                className={cx('searchParams')}
                hotelInfo={hotelInfo}
                searchParams={searchParams}
                showChangeParamsLink={false}
                {...prepareQaAttributes({
                    current: 'bookSearchParams',
                })}
            />
        );
    }

    private renderRoomInfo(): ReactNode {
        const {roomInfo, deviceType} = this.props;

        return (
            <BookRoomDescription
                className={cx('roomDescription')}
                roomInfo={roomInfo}
                deviceType={deviceType}
                showDescription={false}
            />
        );
    }

    private renderBedsGroups(): ReactNode {
        const {
            bedsGroups,
            setBedsGroupId,
            bookUserSelect: {bedsGroupId},
        } = this.props;

        return (
            <BookSelectBedsGroups
                className={cx('bedsGroups')}
                setBedsGroupId={setBedsGroupId}
                bedsGroups={bedsGroups}
                bedsGroupId={bedsGroupId}
            />
        );
    }

    private renderMealInfo(): ReactNode {
        const {mealInfo} = this.props;

        if (!mealInfo) {
            return null;
        }

        return <BookMealInfo className={cx('mealInfo')} mealInfo={mealInfo} />;
    }

    private renderSettlementInfo(): ReactNode {
        const {
            hotelInfo: {settlementInfo},
            deviceType: {isMobile},
        } = this.props;

        return (
            <BookSettlementInfo
                isMobile={isMobile}
                className={cx('settlementInfo')}
                settlementInfo={settlementInfo}
            />
        );
    }

    private renderHotelImage(): ReactNode {
        const {
            hotelInfo: {imageUrlTemplate = ''},
            deviceType: {isMobile},
        } = this.props;

        const preparedImageUrl = parseImageUrlTemplate(
            imageUrlTemplate,
            isMobile
                ? EHotelImageAltayStandardSize.S
                : EHotelImageAltayStandardSize.M,
            //костыль, обязательно уберем в https://st.yandex-team.ru/HOTELS-5834, нужно добавить размеры в ручку
            null,
        );

        return (
            <Image
                className={cx('hotelImageContainer')}
                imageClassName={cx('hotelImage')}
                onClick={this.handleClickPartnerModalTrigger}
                isCrop={true}
                isWide={true}
                src={preparedImageUrl}
                {...prepareQaAttributes({
                    current: 'hotelImage',
                })}
            />
        );
    }

    private renderRoomAmenities(): ReactNode {
        const {
            roomInfo: {amenities},
        } = this.props;

        return (
            <BookHotelAmenities
                className={cx('roomAmenities')}
                amenities={amenities}
            />
        );
    }

    private renderRefundableRules(): ReactNode {
        const {cancellationInfo} = this.props;

        return (
            <OfferRefundableRules
                className={cx('refundableRules')}
                cancellationInfo={cancellationInfo}
            />
        );
    }

    private renderPartnerHotelInfoLink(): ReactNode {
        return (
            <LinkButton
                theme="normal"
                className={cx('partnerDescriptionLink')}
                onClick={this.handleClickPartnerModalTrigger}
                {...prepareQaAttributes({
                    current: 'partnerDescriptionLink',
                })}
            >
                {i18nBlock.partnerDescription()}
            </LinkButton>
        );
    }

    private renderPartnerHotelInfo(): ReactNode {
        const {
            partnerHotelInfo,
            deviceType: {isMobile},
        } = this.props;
        const {canRenderPartnerInfoModal} = this.state;

        return (
            canRenderPartnerInfoModal && (
                <BookPartnerHotelInfo
                    isMobile={isMobile}
                    isVisible={true}
                    onHideModal={this.hidePartnerInfoModal}
                    partnerHotelInfo={partnerHotelInfo}
                />
            )
        );
    }

    private renderPartnerHotelName(): ReactNode {
        const {
            hotelInfo: {partnerHotelName},
        } = this.props;

        return (
            Boolean(partnerHotelName) && (
                <div
                    className={cx('partnerHotelName')}
                    {...prepareQaAttributes({
                        current: 'partnerHotelName',
                    })}
                >
                    {i18nBlock.partnerHotelName({
                        hotelName: partnerHotelName,
                    })}
                </div>
            )
        );
    }

    private renderHotelAndOfferRoomInfo(): ReactNode {
        const {deviceType} = this.props;

        return (
            <TwoColumnLayout
                className={cx('content')}
                deviceType={deviceType}
                rightColumnOffset={5}
                rightColumnWidth={50}
            >
                <TwoColumnLayout.LeftColumn className={cx('leftColumn')}>
                    {this.renderHotelBaseInfo()}
                    {this.renderSearchParams()}
                    {this.renderRoomInfo()}
                    {this.renderRoomAmenities()}
                    {this.renderMealInfo()}
                    {this.renderRefundableRules()}
                    {this.renderBedsGroups()}
                    {this.renderSettlementInfo()}
                </TwoColumnLayout.LeftColumn>
                <TwoColumnLayout.RightColumn className={cx('rightColumn')}>
                    {this.renderHotelImage()}
                    {this.renderPartnerHotelName()}
                    {this.renderPartnerHotelInfoLink()}
                    {this.renderPartnerHotelInfo()}
                </TwoColumnLayout.RightColumn>
            </TwoColumnLayout>
        );
    }

    private renderHotelAndExpiredOfferRoomInfo(): ReactNode {
        const {footerNode, preventOpacityOverlay, deviceType} = this.props;

        return (
            <>
                <TwoColumnLayout
                    className={cx('expiredOfferContent', {
                        expiredOfferContent_preventOpacity:
                            preventOpacityOverlay,
                    })}
                    deviceType={deviceType}
                    rightColumnOffset={5}
                    rightColumnWidth={50}
                >
                    <TwoColumnLayout.LeftColumn className={cx('leftColumn')}>
                        {this.renderHotelBaseInfo()}
                        {this.renderSearchParams()}
                        {this.renderRoomInfo()}
                        {this.renderRoomAmenities()}
                        {this.renderMealInfo()}
                        {this.renderBedsGroups()}
                    </TwoColumnLayout.LeftColumn>
                    <TwoColumnLayout.RightColumn className={cx('rightColumn')}>
                        {this.renderHotelImage()}
                        {this.renderPartnerHotelName()}
                        {this.renderPartnerHotelInfoLink()}
                        {this.renderPartnerHotelInfo()}
                    </TwoColumnLayout.RightColumn>
                </TwoColumnLayout>
                <div className={cx('footer')}>{footerNode}</div>
            </>
        );
    }

    render(): ReactNode {
        const {className, isExpiredOffer, deviceType} = this.props;

        return (
            <div
                className={cx(
                    'bookHotelInfo',
                    deviceMods('bookHotelInfo', deviceType),
                    className,
                )}
                {...prepareQaAttributes({
                    current: 'bookHotelInfo',
                })}
            >
                {isExpiredOffer
                    ? this.renderHotelAndExpiredOfferRoomInfo()
                    : this.renderHotelAndOfferRoomInfo()}
            </div>
        );
    }
}
export default BookHotelInfo;
