import {PureComponent, ReactNode} from 'react';

import {IWithClassName} from 'types/withClassName';
import {
    BookLegalInfoType,
    IBookLegalInfo,
} from 'server/api/HotelsBookAPI/types/IBookOffer';

import * as i18nBlock from 'i18n/hotels-BookLicenseAgreement';

import ModalWithHistoryBack from 'containers/withSupportHistoryBack/ModalWithHistoryBack/ModalWithHistoryBack';

import Link from 'components/Link/Link';
import Modal from 'components/Modal/Modal';
import LinkButton from 'components/LinkButton/LinkButton';

import cx from './BookLicenseAgreement.scss';

const licenseAgreementsLinks = [
    {
        label: i18nBlock.userAgreement(),
        link: i18nBlock.userAgreementLink(),
    },
    {
        label: i18nBlock.userInfoAgreement(),
        link: i18nBlock.userInfoAgreementLink(),
    },
];

interface IBookLicenseAgreementProps extends IWithClassName {
    isMobile: boolean;
    isBookStep: boolean;
    legalInfo: BookLegalInfoType;
}

interface IBookLicenseAgreementState {
    partnerAgreementsIsVisible: boolean;
}

class BookLicenseAgreement extends PureComponent<
    IBookLicenseAgreementProps,
    IBookLicenseAgreementState
> {
    static defaultProps = {
        legalInfo: [],
        isMobile: false,
        isBookStep: true,
    };

    state: IBookLicenseAgreementState = {
        partnerAgreementsIsVisible: false,
    };

    openPartnerAgreements = (): void => {
        this.setState({partnerAgreementsIsVisible: true});
    };
    closePartnerAgreements = (): void => {
        this.setState({partnerAgreementsIsVisible: false});
    };

    renderLicenseAgreements(): ReactNode {
        return (
            <div className={cx('textBlock')}>
                {i18nBlock.agreementLabel()}&nbsp;
                {licenseAgreementsLinks.map(({label, link}, index) => (
                    <span key={index}>
                        <Link
                            url={link}
                            theme="normal"
                            target="_blank"
                            text={label}
                        />
                        {licenseAgreementsLinks.length === index + 1
                            ? '.'
                            : ', '}
                    </span>
                ))}
            </div>
        );
    }

    renderPayLabel(): ReactNode {
        return (
            <div className={cx('textBlock')}>
                {i18nBlock.prepayChargeTitle()}
            </div>
        );
    }

    renderPartnerAgreementsModal = (): ReactNode => {
        const {legalInfo} = this.props;

        return (
            <Modal.Content>
                <div className={cx('partnersTitle')}>
                    {i18nBlock.partnersRequisites()}
                </div>
                <div className={cx('partnersReference')}>
                    {i18nBlock.partnersReferenceInformation()}
                </div>
                {legalInfo.map(this.renderPartnerAgreement)}
            </Modal.Content>
        );
    };

    renderPartnerAgreement = (
        {
            name,
            ogrn,
            actualAddress,
            legalAddress,
            workingHours,
            registryNumber,
        }: IBookLegalInfo,
        index: number,
    ): ReactNode =>
        Boolean(
            name &&
                (ogrn ||
                    actualAddress ||
                    legalAddress ||
                    workingHours ||
                    registryNumber),
        ) && (
            <div className={cx('partnerAgreement')} key={index}>
                <div className={cx('partnerName')}>{name}</div>
                <table>
                    <tbody>
                        {ogrn && (
                            <tr>
                                <td className={cx('partnerAgreementLabel')}>
                                    {i18nBlock.partnersOgrn()}
                                </td>
                                <td>{ogrn}</td>
                            </tr>
                        )}
                        {registryNumber && (
                            <tr>
                                <td className={cx('partnerAgreementLabel')}>
                                    {i18nBlock.partnersRegistryNumber()}
                                </td>
                                <td>{registryNumber}</td>
                            </tr>
                        )}
                        {actualAddress && (
                            <tr>
                                <td className={cx('partnerAgreementLabel')}>
                                    {i18nBlock.partnersActualAddress()}
                                </td>
                                <td>{actualAddress}</td>
                            </tr>
                        )}
                        {legalAddress && (
                            <tr>
                                <td className={cx('partnerAgreementLabel')}>
                                    {i18nBlock.partnersLegalAddress()}
                                </td>
                                <td>{legalAddress}</td>
                            </tr>
                        )}
                        {workingHours && (
                            <tr>
                                <td className={cx('partnerAgreementLabel')}>
                                    {i18nBlock.partnersWorkingHours()}
                                </td>
                                <td>{workingHours}</td>
                            </tr>
                        )}
                    </tbody>
                </table>
            </div>
        );

    renderPartnerAgreements(): ReactNode {
        const {isMobile} = this.props;
        const {partnerAgreementsIsVisible} = this.state;

        return (
            <span className={cx('textBlock')}>
                {i18nBlock.hotelInfoSource()}&nbsp;
                <LinkButton theme="normal" onClick={this.openPartnerAgreements}>
                    {i18nBlock.partnersInfoLink()}
                </LinkButton>
                <ModalWithHistoryBack
                    isMobile={isMobile}
                    isVisible={partnerAgreementsIsVisible}
                    onClose={this.closePartnerAgreements}
                >
                    {this.renderPartnerAgreementsModal()}
                </ModalWithHistoryBack>
            </span>
        );
    }

    renderBookStepAgreement(): ReactNode {
        return (
            <>
                {this.renderLicenseAgreements()}
                {this.renderPayLabel()}
            </>
        );
    }

    render(): ReactNode {
        const {className, isBookStep} = this.props;

        return (
            <div className={className}>
                {isBookStep && this.renderBookStepAgreement()}
                {this.renderPartnerAgreements()}
            </div>
        );
    }
}

export default BookLicenseAgreement;
