import {PureComponent, ReactNode} from 'react';

import {
    IBookHotelCoordinates,
    IBookPartnerHotelDescriptions,
    IBookPartnerHotelInfo,
} from 'server/api/HotelsBookAPI/types/IBookOffer';

import {getImagesBySize} from 'projects/depreacted/hotels/utilities/getImagesBySize/getImagesBySize';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-BookPartnerHotelInfo';

import ModalWithHistoryBack from 'containers/withSupportHistoryBack/ModalWithHistoryBack/ModalWithHistoryBack';

import YandexMaps from 'components/YandexMaps/YandexMaps';
import HotelAddress from 'projects/depreacted/hotels/components/HotelAddress/HotelAddress';
import Modal from 'components/Modal/Modal';
import ImagesCarousel from 'components/ImagesCarousel/ImagesCarousel';

import BookPartnerHotelAmenities from './blocks/BookPartnerHotelAmenities/BookPartnerHotelAmenities';

import cx from './BookPartnerHotelInfo.scss';

const HOTEL_DESCRIPTIONS: (keyof IBookPartnerHotelDescriptions)[] = [
    'location',
    'rooms',
    'dining',
    'amenities',
    'businessAmenities',
];

const IMAGE_SIZE = '350px';
const IMAGES_CAROUSEL_HEIGHT = 300;

interface IBookPartnerHotelInfoProps {
    isMobile: boolean;
    isVisible: boolean;
    partnerHotelInfo: IBookPartnerHotelInfo;
    onHideModal: () => void;
}

class BookPartnerHotelInfo extends PureComponent<IBookPartnerHotelInfoProps> {
    static defaultProps = {
        isMobile: false,
        isVisible: false,
    };

    getFullHotelName = (): string => {
        const {
            partnerHotelInfo: {
                hotelName,
                hotelType,
                propertyRating: {type, rating},
            },
        } = this.props;

        // Добавил дополнительную проверку в случае, если пришло 0.0 TRAVELFRONT-3309
        const hotelStarsCount =
            type === 'Star' && rating ? parseInt(rating, 10) : 0;
        const hotelStars = hotelStarsCount ? `${hotelStarsCount}*` : '';

        return [hotelType, hotelName, hotelStars].filter(Boolean).join(' ');
    };

    getFullHotelAddress = (): string => {
        const {
            partnerHotelInfo: {
                address: {city = '', countryCode = '', firstLine, secondLine},
            },
        } = this.props;

        return [countryCode, city, firstLine, secondLine]
            .filter(Boolean)
            .join(', ');
    };

    renderHotelName(): ReactNode {
        return (
            <h2
                className={cx('hotelName')}
                {...prepareQaAttributes({current: 'hotelName'})}
            >
                {this.getFullHotelName()}
            </h2>
        );
    }

    renderHotelAddress(): ReactNode {
        return (
            <HotelAddress
                className={cx('hotelAddress')}
                text={this.getFullHotelAddress()}
            />
        );
    }

    renderYandexMap(coordinates: IBookHotelCoordinates): ReactNode {
        const {lat, lng: lon} = coordinates;

        return (
            Boolean(lat && lon) && (
                <div
                    className={cx('mapContainer')}
                    {...prepareQaAttributes({
                        current: 'mapContainer',
                    })}
                >
                    <YandexMaps
                        className={cx('map')}
                        defaultCenter={{lat, lon}}
                        mapMarkers={[{lat, lon}]}
                        controls={[]}
                    />
                </div>
            )
        );
    }

    renderCarouselWithImagesAndMap(): ReactNode {
        const {
            partnerHotelInfo: {images, coordinates},
            isMobile,
        } = this.props;

        const preparedImages = getImagesBySize(images, IMAGE_SIZE);
        const YandexMapNode = this.renderYandexMap(coordinates);

        return (
            <ImagesCarousel
                className={cx('imagesCarousel')}
                arrowsType="dark"
                images={preparedImages}
                spaceBetween={0}
                beforeNode={YandexMapNode}
                hideArrows={false}
                isDesktop={!isMobile}
                height={IMAGES_CAROUSEL_HEIGHT}
            />
        );
    }

    renderHotelDescriptions(): ReactNode {
        const {
            partnerHotelInfo: {descriptions},
        } = this.props;

        return (
            <div
                className={cx('hotelDescriptions')}
                {...prepareQaAttributes({
                    current: 'hotelDescriptions',
                })}
            >
                {HOTEL_DESCRIPTIONS.map(
                    descriptionType =>
                        Boolean(descriptions[descriptionType]) && (
                            <section
                                className={cx('hotelDescription')}
                                key={descriptionType}
                            >
                                <h5 className={cx('hotelDescriptionTitle')}>
                                    {i18nBlock[descriptionType]()}
                                </h5>
                                {descriptions[descriptionType]}
                            </section>
                        ),
                )}
            </div>
        );
    }

    renderHotelAmenities(): ReactNode {
        const {
            partnerHotelInfo: {amenities},
        } = this.props;

        return (
            <BookPartnerHotelAmenities
                className={cx('hotelAmenities')}
                amenities={amenities}
            />
        );
    }

    renderPartnerTitle(): ReactNode {
        return (
            <div className={cx('partnerTitle')}>{i18nBlock.partnerTitle()}</div>
        );
    }

    renderPartnerHotelInfo = (): ReactNode => {
        const {isMobile} = this.props;

        return (
            <Modal.Content
                inset="0"
                className={cx('partnerHotelInfo', {
                    partnerHotelInfo_mobile: isMobile,
                })}
            >
                {this.renderHotelName()}
                {this.renderHotelAddress()}
                {this.renderCarouselWithImagesAndMap()}
                {this.renderHotelDescriptions()}
                {this.renderHotelAmenities()}
                {this.renderPartnerTitle()}
            </Modal.Content>
        );
    };

    render(): ReactNode {
        const {isVisible, onHideModal, isMobile} = this.props;

        return (
            <ModalWithHistoryBack
                isMobile={isMobile}
                isVisible={isVisible}
                onClose={onHideModal}
            >
                {this.renderPartnerHotelInfo()}
            </ModalWithHistoryBack>
        );
    }
}

export default BookPartnerHotelInfo;
