import React, {useRef, useMemo, ReactElement, useCallback} from 'react';

import {ENTER} from 'constants/eventKeys';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {
    EPromoCodesApplicationType,
    IPromoCodeApplicationResult,
} from 'types/hotels/book/IApplyPromoCodes';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {IBookPromoCodesReducer} from 'reducers/depreacted/hotels/bookAndPayPage/promoCodesInfo/reducer';
import {TUserInfo} from 'reducers/common/commonReducerTypes';

import {deviceModDesktop} from 'utilities/stylesUtils';
import {CHAR_MINUS, CHAR_THIN_SPACE} from 'utilities/strings/charCodes';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';

import * as i18nBlock from 'i18n/hotels-BookPriceInfo';

import Input from 'components/Input/Input';
import Price from 'components/Price/Price';
import Button from 'components/Button/Button';
import Spinner from 'components/Spinner/Spinner';
import LinkButton from 'components/LinkButton/LinkButton';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';
import IncludeIcon from 'icons/16/Include';
import PromoCodeError from 'projects/depreacted/hotels/components/PromoCodeError/PromoCodeError';

import cx from './InputControls.scss';

const MOBILE_TOOLTIP_DIRECTIONS = [
    EPopupDirection.TOP_RIGHT,
    EPopupDirection.TOP,
];
const DESKTOP_TOOLTIP_DIRECTIONS = [EPopupDirection.LEFT];

interface IInputControlsProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    userInfo: TUserInfo;
    promoCodesInfo: IBookPromoCodesReducer;
    appliedPromoCode?: IPromoCodeApplicationResult;
    inputValue: string;

    onInputChange: (value: string) => void;
    onApplyClick: (promoCodes: string[]) => void;
    onResetClick: () => void;
}

const InputControls: React.FC<IInputControlsProps> = ({
    className,
    deviceType,
    userInfo,
    promoCodesInfo,
    inputValue,
    appliedPromoCode,
    onInputChange,
    onApplyClick,
    onResetClick,
    ...props
}) => {
    const {isMobile} = deviceType;
    const {isLoading, isFailed, isSuccess} = promoCodesInfo;

    const isEmptyInput = inputValue.trim().length === 0;
    const isSuccessCode =
        appliedPromoCode?.type === EPromoCodesApplicationType.SUCCESS;
    const isApplySuccess = isSuccess && isSuccessCode;
    const isDisabledControls = isLoading || isApplySuccess;
    const isDisabled = !isAuthUser(userInfo) || isDisabledControls;
    const isApplyError = isFailed || (isSuccess && !isSuccessCode);

    const inputContainerRef = useRef<HTMLDivElement>(null);

    const {
        value: canShowError,
        setTrue: enableError,
        setFalse: disableError,
    } = useBoolean(true);

    const iconRight = useMemo((): ReactElement | undefined => {
        switch (true) {
            case isLoading:
                return <Spinner size="xxs" />;
            case isApplySuccess:
                return <IncludeIcon className={cx('checkIcon')} />;
        }

        return undefined;
    }, [isLoading, isApplySuccess]);

    const handleApplyButtonClick = useCallback(() => {
        const value = inputValue.trim().toUpperCase();

        onInputChange(value);
        onApplyClick([value]);
        enableError();
    }, [inputValue, onInputChange, onApplyClick, enableError]);

    const handleInputChange = useCallback(
        (_: unknown, value: string = '') => {
            onInputChange(value);
            disableError();
        },
        [onInputChange, disableError],
    );

    const handleKeyDown = useCallback(
        (event: React.KeyboardEvent) => {
            if (event.key === ENTER) {
                handleApplyButtonClick();
            }
        },
        [handleApplyButtonClick],
    );

    const handleInputRef = useCallback(ref => {
        if (ref) {
            ref.focus();
        }
    }, []);

    return (
        <>
            <div className={cx('root', className)}>
                <div className={cx('inputContainer')} ref={inputContainerRef}>
                    <Input
                        className={cx('input')}
                        size={isMobile ? 'l' : 'm'}
                        disabled={isDisabled}
                        iconRight={iconRight}
                        value={inputValue}
                        state={isApplyError ? 'error' : undefined}
                        onChange={handleInputChange}
                        onKeyDown={handleKeyDown}
                        inputRef={handleInputRef}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'input',
                        })}
                    />
                </div>
                <div>
                    {isApplySuccess && appliedPromoCode?.discountAmount && (
                        <Price
                            className={cx('discountPrice')}
                            prefix={CHAR_MINUS}
                            prefixDelimiter={CHAR_THIN_SPACE}
                            {...appliedPromoCode.discountAmount}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'discountAmount',
                            })}
                        />
                    )}
                    <Button
                        className={cx({
                            button_hidden: isApplySuccess,
                        })}
                        size={isMobile ? 'l' : 'm'}
                        disabled={isDisabled || isEmptyInput}
                        onClick={handleApplyButtonClick}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'button',
                        })}
                    >
                        {i18nBlock.promocodeDotApply()}
                    </Button>
                </div>
            </div>
            {isApplySuccess && (
                <LinkButton
                    className={cx('resetLink')}
                    onClick={onResetClick}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'reset',
                    })}
                >
                    {i18nBlock.promocodeDotReset()}
                </LinkButton>
            )}
            <MessageBoxPopup
                direction={
                    isMobile
                        ? MOBILE_TOOLTIP_DIRECTIONS
                        : DESKTOP_TOOLTIP_DIRECTIONS
                }
                anchorRef={inputContainerRef}
                isVisible={canShowError && isApplyError}
                onClose={disableError}
            >
                <div
                    className={cx(
                        'tooltipContent',
                        deviceModDesktop('tooltipContent', deviceType),
                    )}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'tooltip',
                    })}
                >
                    <PromoCodeError status={appliedPromoCode?.type} />
                </div>
            </MessageBoxPopup>
        </>
    );
};

export default InputControls;
