import {FunctionComponent, useState} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {IBookOfferRoom} from 'server/api/HotelsBookAPI/types/IBookOffer';

import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-BookRoomDescription';

import Text from 'components/Text/Text';
import Heading from 'components/Heading/Heading';
import LinkButton from 'components/LinkButton/LinkButton';
import AutoHeightAnimateControl from 'components/AutoHeightAnimateControl/AutoHeightAnimateControl';
import PartnerRichContent from 'projects/depreacted/hotels/components/PartnerRichContent/PartnerRichContent';
import RotatingArrowIcon from 'components/RotatingArrowIcon/RotatingArrowIcon';

import cx from './BookRoomDescription.scss';

export interface IBookRoomDescriptionProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    roomInfo: IBookOfferRoom;
    showDescription?: boolean;
}

const BookRoomDescription: FunctionComponent<IBookRoomDescriptionProps> =
    props => {
        const {
            className,
            showDescription = true,
            deviceType,
            roomInfo: {
                name,
                descriptions: {overview},
            },
        } = props;
        const [showFullDescription, toggleFullDescriptionView] =
            useState(false);

        return (
            <div
                className={cx(
                    'content',
                    deviceMods('content', deviceType),
                    className,
                )}
            >
                <Heading
                    className={cx('roomName')}
                    level={3}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'offer-name',
                    })}
                >
                    {name}
                </Heading>
                {typeof overview !== 'undefined' &&
                    overview !== '' &&
                    showDescription && (
                        <>
                            <Text size="m">
                                <LinkButton
                                    theme="normal"
                                    className={cx('descriptionToggler')}
                                    onClick={(): void => {
                                        toggleFullDescriptionView(
                                            !showFullDescription,
                                        );
                                    }}
                                >
                                    {i18nBlock.showFullDescription()}
                                    <RotatingArrowIcon
                                        className={cx('descriptionToggleIcon')}
                                        rotated={showFullDescription}
                                    />
                                </LinkButton>
                            </Text>
                            <AutoHeightAnimateControl
                                canShowComponent={showFullDescription}
                            >
                                <PartnerRichContent
                                    content={overview}
                                    size="m"
                                />
                            </AutoHeightAnimateControl>
                        </>
                    )}
            </div>
        );
    };

export default BookRoomDescription;
