import {PureComponent, ReactNode} from 'react';

import {TRAVELLINE_PARTNER_ID, BNOVO_PARTNER_ID} from 'constants/hotels';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-BookSettlementInfo';

import LinkButton from 'components/LinkButton/LinkButton';
import Separator from 'components/Separator/Separator';
import AutoHeightAnimateControl from 'components/AutoHeightAnimateControl/AutoHeightAnimateControl';
import PartnerRichContent from 'projects/depreacted/hotels/components/PartnerRichContent/PartnerRichContent';
import RotatingArrowIcon from 'components/RotatingArrowIcon/RotatingArrowIcon';
import ArrowLeftIcon from 'icons/16/ArrowLeft';

import cx from './BookSettlementInfo.scss';

const DEFAULT_FULL_RENDER_SETTLEMENT_PARTNER_IDS = [
    TRAVELLINE_PARTNER_ID,
    BNOVO_PARTNER_ID,
];

interface IBookSettlementInfoProps extends IWithQaAttributes {
    className?: string;
    // Для тачей, текст на кнопке назад в открытом модале
    backName?: string;
    isMobile: boolean;
    settlementInfo: any;
    partnerId?: string;
}

interface IBookSettlementInfoState {
    canRenderAllInfo: boolean;
}

class BookSettlementInfo extends PureComponent<
    IBookSettlementInfoProps,
    IBookSettlementInfoState
> {
    state = {
        canRenderAllInfo: false,
    };

    private toggleSettlementView = (): void => {
        this.setState(({canRenderAllInfo}) => ({
            canRenderAllInfo: !canRenderAllInfo,
        }));
    };

    private getSettlementInfo(): string[] {
        const {
            settlementInfo: {
                instructions,
                optionalCharge,
                mandatoryCharge,
                policyCheckinInfo,
                specialInstructions,
            },
        } = this.props;

        return [
            mandatoryCharge,
            specialInstructions,
            instructions,
            optionalCharge,
            policyCheckinInfo,
        ].filter(Boolean);
    }

    private renderSettlementItems = (items: string[]): ReactNode =>
        items.map((settlementText, index) => (
            <PartnerRichContent key={index} size="m" content={settlementText} />
        ));

    private renderSettlementInfo(): ReactNode {
        const {canRenderAllInfo} = this.state;
        const settlementInfoItems = this.getSettlementInfo();

        return (
            <AutoHeightAnimateControl canShowComponent={canRenderAllInfo}>
                <div className={cx('settlementInfoContent')}>
                    {this.renderSettlementItems(settlementInfoItems)}
                </div>
            </AutoHeightAnimateControl>
        );
    }

    private renderSettlementModal(): ReactNode {
        return null;
    }

    private renderTitle(): ReactNode {
        const {canRenderAllInfo} = this.state;
        const {isMobile} = this.props;

        return (
            <h5
                onClick={this.toggleSettlementView}
                className={cx('settlementInfoTitle')}
            >
                <LinkButton
                    theme={isMobile ? 'black' : 'normal'}
                    className={cx({link: isMobile})}
                >
                    {i18nBlock.title()}
                    {!isMobile && (
                        <RotatingArrowIcon
                            className={cx('arrowIcon')}
                            rotated={canRenderAllInfo}
                        />
                    )}
                    {isMobile && <ArrowLeftIcon className={cx('arrowIcon')} />}
                </LinkButton>
            </h5>
        );
    }

    renderContent(): ReactNode {
        const {isMobile, partnerId} = this.props;
        const settlementInfo = this.getSettlementInfo();
        const canDefaultRenderSettlementInfo = partnerId
            ? DEFAULT_FULL_RENDER_SETTLEMENT_PARTNER_IDS.includes(partnerId)
            : false;

        if (canDefaultRenderSettlementInfo) {
            return this.renderSettlementItems(settlementInfo);
        }

        if (isMobile) {
            return (
                <>
                    <Separator margin={3} />
                    {this.renderTitle()}
                    {this.renderSettlementModal()}
                </>
            );
        }

        return (
            <>
                {this.renderTitle()}
                {this.renderSettlementInfo()}
            </>
        );
    }

    render(): ReactNode {
        const {className} = this.props;
        const settlementInfo = this.getSettlementInfo();

        return (
            Boolean(settlementInfo.length) && (
                <section
                    className={cx('settlementInfo', className)}
                    {...prepareQaAttributes(this.props)}
                >
                    {this.renderContent()}
                </section>
            )
        );
    }
}

export default BookSettlementInfo;
