import React, {PureComponent, useRef, useState} from 'react';
import ReactIFrame from 'react-iframe';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-FiscalReceipt';

import ModalWithTrigger from 'components/ModalWithTrigger/ModalWithTrigger';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';

import cx from './FiscalReceipt.scss';

interface IEmptyFiscalTooltipProps {
    triggerText: string;
    triggerClassName?: string;
}

const EmptyFiscalTooltip: React.FC<IEmptyFiscalTooltipProps> = props => {
    const {triggerText, triggerClassName} = props;
    const [canRenderTooltip, setTooltipView] = useState(false);
    const triggerRef = useRef(null);

    return (
        <>
            <span
                className={cx('fiscalReceiptTrigger', triggerClassName)}
                onClick={() => setTooltipView(true)}
                ref={triggerRef}
            >
                {triggerText}
            </span>
            <MessageBoxPopup
                anchorRef={triggerRef}
                isVisible={canRenderTooltip}
                onClose={(): void => setTooltipView(false)}
            >
                <div className={cx('emptyFiscalUrlTooltip')}>
                    {i18nBlock.emptyFiscalReceiptTitle()}
                </div>
            </MessageBoxPopup>
        </>
    );
};

interface IFiscalReceiptProps extends IWithClassName, IWithQaAttributes {
    isMobile: boolean;
    canRenderTooltipForEmptyUrl: boolean;
    triggerText: string;
    fiscalReceipt: {
        url: string;
        type: string;
    };
}

class FiscalReceipt extends PureComponent<IFiscalReceiptProps> {
    static defaultProps = {
        className: '',
        fiscalReceipt: {},
        isMobile: false,
        triggerText: '',
        canRenderTooltipForEmptyUrl: false,
    };

    renderFiscalReceipt = (url: string): React.ReactNode => {
        return <ReactIFrame url={url} className={cx('iframeInstance')} />;
    };

    render(): React.ReactNode {
        const {
            className,
            triggerText,
            fiscalReceipt: {url},
            canRenderTooltipForEmptyUrl,
        } = this.props;

        if (url) {
            return (
                <ModalWithTrigger
                    triggerText={triggerText || i18nBlock.fiscalReceiptTitle()}
                    triggerClassName={cx('fiscalReceiptTrigger', className)}
                    {...prepareQaAttributes(this.props)}
                >
                    <div className={cx('fiscalReceiptModal')}>
                        {this.renderFiscalReceipt(url)}
                    </div>
                </ModalWithTrigger>
            );
        } else if (canRenderTooltipForEmptyUrl) {
            return (
                <EmptyFiscalTooltip
                    triggerText={triggerText}
                    triggerClassName={className}
                />
            );
        }

        return (
            <h5 className={cx('fiscalReceiptTitle', className)}>
                {i18nBlock.emptyFiscalReceiptTitle()}
            </h5>
        );
    }
}

export default FiscalReceipt;
