import {FunctionComponent, useState, useEffect, useCallback} from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';

import {getUserInfo} from 'selectors/common/userInfoSelector';

import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {
    restoreGreetingClosed,
    storeGreetingClosed,
} from './utilities/canRenderGreeting';

import Skeleton from 'components/Skeletons/Skeleton/Skeleton';
import LoginAndSaveMoney from './components/LoginAndSaveMoney/LoginAndSaveMoney';
import ForceLoginGreeting from './components/ForceLoginGreeting/ForceLoginGreeting';

import cx from './ForceLogin.scss';

interface IForceLoginProps extends IWithClassName {
    isLoading?: boolean;
    isBookPage?: boolean;
    isSearchPage?: boolean;
    hasBoyOffers?: boolean;
}

const ForceLogin: FunctionComponent<IForceLoginProps> = ({
    className,
    isLoading,
    isBookPage,
    isSearchPage,
    hasBoyOffers,
}) => {
    const restoredIsGreetingClosed = restoreGreetingClosed();

    const userInfo = useSelector(getUserInfo);
    const [canRender, setCanRender] = useState(false);
    const [isGreetingClosed, setGreetingClosed] = useState(
        restoredIsGreetingClosed,
    );

    const handleCloseGreetingClick = useCallback(() => {
        setGreetingClosed(true);
        storeGreetingClosed(true);
    }, [setGreetingClosed]);

    useEffect(() => {
        if (!canRender) {
            setCanRender(true);
        }
    }, [canRender]);

    if (!canRender || isLoading) {
        return <Skeleton className={cx('skeleton', className)} withAnimation />;
    }

    if (!isAuthUser(userInfo)) {
        return (
            <LoginAndSaveMoney
                className={className}
                isBookPage={isBookPage}
                isSearchPage={isSearchPage}
                hasBoyOffers={hasBoyOffers}
            />
        );
    }

    if (!isGreetingClosed) {
        return (
            <ForceLoginGreeting
                className={className}
                onCloseClick={handleCloseGreetingClick}
                userInfo={userInfo}
                isBookPage={isBookPage}
            />
        );
    }

    return null;
};

export default ForceLogin;
