import {FunctionComponent, useMemo} from 'react';
import moment from 'moment';

import {IWithClassName} from 'types/withClassName';

import {IAuthUserInfo} from 'reducers/common/commonReducerTypes';

import getAvatarUrl from 'utilities/avatar/getAvatarUrl';
import {DEFAULT_AVATAR_ID} from 'utilities/avatar/getAvatarId';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {getGreetingWithDayTime} from 'utilities/greeting/getGreetingWithDayTime';
import getCommonCardProps from 'projects/depreacted/hotels/components/ForceLogin/utilities/getCommonCardProps';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18n/hotels-ForceLogin';

import CloseIcon from 'icons/16/Close';
import Avatar from 'components/Avatar/Avatar';
import Card from 'components/Card/Card';
import Link from 'components/Link/Link';
import ForceLoginLayout from 'components/ForceLogin/ForceLoginLayout/ForceLoginLayout';

interface IForceLoginGreetingProps extends IWithClassName {
    userInfo: IAuthUserInfo;
    isBookPage?: boolean;
    onCloseClick: () => void;
}

const ForceLoginGreeting: FunctionComponent<IForceLoginGreetingProps> = ({
    userInfo,
    isBookPage,
    onCloseClick,
    className,
}) => {
    const {isMobile} = useDeviceType();
    const {displayName: userName, avatarPath, avatar, havePlus} = userInfo;

    const greetingText = useMemo(() => {
        const greetingWithDayTime = getGreetingWithDayTime(moment().hours());

        if (isBookPage) {
            if (havePlus) {
                return i18nBlock.greetingAtBookPageForPlusUser({
                    greetingWithDayTime,
                    userName,
                });
            }

            return insertJSXIntoKey(i18nBlock.greetingAtBookPageForcePlus)({
                greetingWithDayTime,
                userName,
                plusLink: (
                    <Link target="_blank" url={'YANDEX_PLUS_URL'}>
                        {i18nBlock.plus()}
                    </Link>
                ),
            });
        }

        return i18nBlock.greeting({greetingWithDayTime, userName});
    }, [havePlus, userName, isBookPage]);

    if (!isAuthUser(userInfo)) {
        return null;
    }

    const avatarUrl = getAvatarUrl(
        avatarPath,
        avatar.default || DEFAULT_AVATAR_ID,
        {retina: true},
    );

    return (
        <Card className={className} {...getCommonCardProps(isMobile)}>
            <ForceLoginLayout
                image={<Avatar src={avatarUrl} size="m" />}
                text={greetingText}
                icon={CloseIcon}
                onIconClick={onCloseClick}
                hoverType="icon"
            />
        </Card>
    );
};

export default ForceLoginGreeting;
