import React, {useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {closeHotelsGeolocationErrorModalAction} from 'reducers/common/hotelsGeolocationErrorModal/actions';

import getHotelsGeolocationError from 'selectors/common/hotelsGeolocationErrorModalSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {useReachGoal} from 'utilities/metrika/useReachGoal';

import * as i18nBlock from 'i18n/hotels-Geodistance';

import Modal from 'components/Modal/Modal';
import Heading from 'components/Heading/Heading';
import Link from 'components/Link/Link';
import Button from 'components/Button/Button';

import cx from './GeolocationError.scss';

interface IGeolocationError {}

const GeolocationError: React.FC<IGeolocationError> = () => {
    const {isOpen} = useSelector(getHotelsGeolocationError);
    const deviceType = useDeviceType();
    const dispatch = useDispatch();

    const howToClick = useReachGoal(
        EHotelsGoal.MC_HOTELS_GEOLOCATION_ERROR_MODAL_HOW_TO,
    );
    const reachErrorModalShow = useReachGoal(
        EHotelsGoal.MC_HOTELS_GEOLOCATION_ERROR_MODAL,
    );

    useEffect(() => {
        if (isOpen) {
            reachErrorModalShow();
        }
    }, [isOpen]);

    const handleClose = () =>
        dispatch(closeHotelsGeolocationErrorModalAction());

    return (
        <Modal
            isMobile
            isVisible={isOpen}
            hasCloseButton={true}
            onClose={handleClose}
        >
            <Modal.Content
                className={cx('container', deviceMods('container', deviceType))}
            >
                <Heading
                    level={'2'}
                    className={cx('header', deviceMods('header', deviceType))}
                >
                    {i18nBlock.geodistanceErrorTitle()}
                </Heading>
                <div className={cx('content')}>
                    {i18nBlock.geodistanceErrorText()}
                </div>
                <div onClick={howToClick} className={cx('link')}>
                    <Link
                        target="_blank"
                        url={
                            'https://yandex.ru/support/common/browsers-settings/geolocation.html'
                        }
                    >
                        {i18nBlock.geodistanceErrorHowToDo()}
                    </Link>
                </div>
                <Button
                    className={cx('button', deviceMods('button', deviceType))}
                    theme="primary"
                    width={deviceType.isMobile ? 'max' : 'auto'}
                    size={deviceType.isMobile ? 'l' : 'm-inset'}
                    onClick={handleClose}
                >
                    {i18nBlock.geodistanceErrorCancel()}
                </Button>
            </Modal.Content>
        </Modal>
    );
};

export default React.memo(GeolocationError);
