import React, {useCallback, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {TImage} from 'types/common/TImage';

import LazyComponent from 'components/LazyComponent/LazyComponent';
import CommonImagesCarousel from 'components/ImagesCarousel/ImagesCarousel';

import cx from './ImagesCarousel.scss';

interface IImagesCarouselInViewProps extends IWithClassName {
    images: TImage[];
    isLazy: boolean;
    hideArrows: boolean;
    onNavigationButtonClick(): void;
}

const ImagesCarousel: React.FC<IImagesCarouselInViewProps> = props => {
    const {className, images, hideArrows, isLazy, onNavigationButtonClick} =
        props;

    const renderPlaceholder = useCallback(
        ref => <div className={cx('placeholder')} ref={ref} />,
        [],
    );

    const renderContent = useCallback(
        () => (
            <CommonImagesCarousel
                className={className}
                type="mini"
                arrowsType="dark"
                images={images}
                spaceBetween={0}
                hideArrows={hideArrows}
                onPrevButtonClick={onNavigationButtonClick}
                onNextButtonClick={onNavigationButtonClick}
                isDesktop
                autoHeight
                lazyLoad
            />
        ),
        [className, images, hideArrows, onNavigationButtonClick],
    );

    const options = useMemo(
        () => ({
            rootMargin: '300px',
            triggerOnce: true,
            skip: !isLazy,
        }),
        [isLazy],
    );

    if (!isLazy) {
        return renderContent();
    }

    return (
        <LazyComponent
            intersectionObserverOptions={options}
            renderPlaceholder={renderPlaceholder}
            renderContent={renderContent}
        />
    );
};

export default React.memo(ImagesCarousel);
