import React, {FunctionComponent, useCallback, useMemo} from 'react';
import {useHistory} from 'react-router-dom';
import _throttle from 'lodash/throttle';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IHotelOffer} from 'types/hotels/offer/IHotelOffer';
import {EPortalSeedReferer} from 'types/hotels/common/IAttribution';
import {EFavoritesGoal} from 'utilities/metrika/types/goals/favorites';

import {reachGoal} from 'utilities/metrika';
import {convertHotelOrRoomImages} from 'projects/depreacted/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {
    getSearchParamsForBaseHotelUrl,
    hotelsURLs,
} from 'projects/depreacted/hotels/utilities/urls';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlockHotelsShortView from 'i18n/hotels-HotelShortView';
import * as i18nBlock from 'i18n/account-Favorites';

/* Components */
import Link from 'components/Link/Link';
import HotelInfoSearchCard, {
    ECardUsagePage,
} from 'projects/depreacted/hotels/components/HotelInfoSearchCard/HotelInfoSearchCard';
import Text from 'components/Text/Text';
import {EHotelsListSkeletonType} from 'projects/depreacted/hotels/components/HotelsList/HotelsList';
import MobileHotelSearchCardOfferSkeleton from 'projects/depreacted/hotels/pages/SearchPage/components/MobileHotelSearchCardOfferSkeleton/MobileHotelSearchCardOfferSkeleton';
import HotelCardLabels from 'projects/depreacted/hotels/pages/SearchPage/components/HotelCardLabels/HotelCardLabels';
import ImagesCarousel from 'projects/depreacted/hotels/components/HotelCard/components/HotelCardMobile/components/ImagesCarousel/ImagesCarousel';
import FavoritesSolid from 'projects/depreacted/hotels/components/Favorites/FavoritesSolid';

import HotelFirstOffer from '../HotelFirstOffer/HotelFirstOffer';
import {IHotelCardProps} from '../../HotelCard';

/* Init styles */
import cx from './HotelCardMobile.scss';

/* Constants */
const HOTEL_IMAGE_SIZE_NAME = 'M';
const CAROUSEL_SCROLL_THROTTLE_TIME = 1000;

const HotelCardMobile: FunctionComponent<IHotelCardProps> = props => {
    const {
        className,
        hotelWithOffers,
        offerRequestParams,
        nights,
        onCardClick,
        isLazyImagesCarousel = false,
        onFavoriteClick,
        renderOfferSkeleton,
        skeletonType,
        currentPage,
    } = props;
    const {hotel, offers, searchedByUser} = hotelWithOffers;
    const {permalink, images} = hotel;
    const [firstOffer]: IHotelOffer[] = offers || [];
    const history = useHistory();
    const experiments = useExperiments();
    const {hotelsPercentDiscount} = experiments;
    const deviceType = useDeviceType();

    const convertedImages = useMemo(
        () => convertHotelOrRoomImages(images, HOTEL_IMAGE_SIZE_NAME),
        [images],
    );

    const handleCarouselContentScroll = useCallback(
        _throttle(
            () => {
                reachGoal(EHotelsGoal.SEARCH_PAGE_IMAGE_SWIPE);

                if (currentPage === 'map') {
                    reachGoal(EHotelsGoal.SEARCHMAP_CARD_IMAGE_SWIPE);
                }

                if (currentPage === 'search') {
                    reachGoal(EHotelsGoal.SEARCHPAGE_CARD_IMAGE_SWIPE);
                }
            },
            CAROUSEL_SCROLL_THROTTLE_TIME,
            {trailing: false},
        ),
        [currentPage],
    );

    const handleCardClick = useCallback(() => {
        onCardClick?.(Boolean(searchedByUser));
    }, [onCardClick, searchedByUser]);

    const handleCardContentClick = useCallback(() => {
        reachGoal(EHotelsGoal.SEARCH_PAGE_CARD_HOTEL_CLICK);

        if (currentPage === 'map') {
            reachGoal(EHotelsGoal.SEARCHMAP_CARD_CLICK);
        }

        if (currentPage === 'search') {
            reachGoal(EHotelsGoal.SEARCHPAGE_CARD_CLICK);
        }

        handleCardClick();
    }, [currentPage, handleCardClick]);
    const searchParams = getSearchParamsForBaseHotelUrl({
        seed: EPortalSeedReferer.PORTAL_HOTELS_SEARCH,
        hotel: hotelWithOffers.hotel,
        offers: hotelWithOffers.offers,
        offerRequestParams,
    });
    const hotelUrl = hotelsURLs.getHotelUrl(searchParams);

    const handleSlideImageClick = useCallback(() => {
        history.push(hotelUrl);
        handleCardContentClick();
    }, [history, hotelUrl, handleCardContentClick]);

    const offer = useMemo(() => {
        if (
            !firstOffer &&
            skeletonType === EHotelsListSkeletonType.OFFER_SKELETON &&
            !renderOfferSkeleton
        ) {
            return (
                <Text size="m" className={cx('noRooms')} weight="medium">
                    {i18nBlock.noRooms()}
                </Text>
            );
        }

        if (
            !firstOffer &&
            skeletonType === EHotelsListSkeletonType.OFFER_SKELETON &&
            renderOfferSkeleton
        ) {
            return <MobileHotelSearchCardOfferSkeleton />;
        }

        return firstOffer ? (
            <HotelFirstOffer
                className={cx('offerInfo')}
                offer={firstOffer}
                firstOfferUrl={hotelUrl}
                nights={nights}
                hotelsPercentDiscount={hotelsPercentDiscount}
                deviceType={deviceType}
            />
        ) : (
            <div className={cx('noOffers')}>
                {i18nBlockHotelsShortView.noOffersNew()}
            </div>
        );
    }, [
        renderOfferSkeleton,
        firstOffer,
        hotelUrl,
        nights,
        skeletonType,
        hotelsPercentDiscount,
        deviceType,
    ]);

    const imagesCarousel = useMemo(() => {
        if (!convertedImages?.length) return null;

        return (
            <ImagesCarousel
                images={convertedImages}
                isLazy={isLazyImagesCarousel}
                onScrollContent={handleCarouselContentScroll}
                onImageClick={handleSlideImageClick}
            />
        );
    }, [
        convertedImages,
        handleCarouselContentScroll,
        handleSlideImageClick,
        isLazyImagesCarousel,
    ]);

    const handleFavoriteClick = useCallback(() => {
        if (onFavoriteClick) {
            onFavoriteClick({permalink, isFavorite: hotel.isFavorite});
            reachGoal(EFavoritesGoal.FAVORITES_CLICK);

            if (currentPage === 'map') {
                reachGoal(EHotelsGoal.SEARCHMAP_CARD_FAV);
            }

            if (currentPage === 'search') {
                reachGoal(EHotelsGoal.SEARCHPAGE_CARD_FAV);
            }
        }
    }, [onFavoriteClick, permalink, hotel.isFavorite, currentPage]);

    return (
        <section
            key={permalink}
            className={cx('mobileHotelCard', className)}
            {...prepareQaAttributes({
                current: props,
                key: permalink,
            })}
        >
            {imagesCarousel}

            <div className={cx('heartIcon')}>
                <FavoritesSolid
                    size="m"
                    onClick={handleFavoriteClick}
                    isFavorite={hotel.isFavorite}
                />
            </div>

            {experiments.hotelsTopSnippet && (
                <HotelCardLabels
                    className={cx('labels')}
                    searchedByUser={searchedByUser}
                />
            )}

            <div className={cx('content')}>
                <Link
                    to={hotelUrl}
                    className={cx('hotelLink')}
                    onClick={handleCardContentClick}
                    rel="noopener noreferrer"
                />
                <HotelInfoSearchCard
                    className={cx('hotelInfo')}
                    hotel={hotel}
                    hotelUrl={hotelUrl}
                    hotelReviewsUrl={hotelUrl}
                    cardUsagePage={ECardUsagePage.SEARCH_PAGE}
                    onFavoriteClick={handleFavoriteClick}
                />
                {offer}
            </div>
        </section>
    );
};

export default React.memo(HotelCardMobile);
