import React, {FunctionComponent, useCallback, useMemo, useState} from 'react';

/* Utilities */
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {convertHotelOrRoomImages} from 'projects/depreacted/hotels/utilities/prepareAndParseImages/prepareAndParseImages';

import * as i18nBlock from 'i18n/hotels-HotelPageRooms';

/* Components */
import Link from 'components/Link/Link';
import ImagesCarousel from 'components/ImagesCarousel/ImagesCarousel';
import HotelImageStub from 'projects/depreacted/hotels/components/HotelImageStub/HotelImageStub';
import HotelInfoSearchCard from 'projects/depreacted/hotels/components/HotelInfoSearchCard/HotelInfoSearchCard';

import HotelMinPriceOffer from '../HotelMinPriceOffer/HotelMinPriceOffer';
import {ICommonHotelCardWithMinPriceProps} from '../../HotelCardWithMinPrice';

/* Init styles */
import cx from './HotelCardWithMinPriceDesktop.scss';

/* Constants */
const HOTEL_IMAGE_SIZE_NAME = 'L';
const BUILDING_ILLUSTRATION_PROPS = {height: 64, width: 64};

const HotelCardWithMinPriceDesktop: FunctionComponent<ICommonHotelCardWithMinPriceProps> =
    props => {
        const [isHovered, onHoverHotel] = useState<boolean>(false);
        const {
            className,
            cardUsagePage,
            hotelInfoClassName,
            hotelWithMinPrice,
            hotelUrl,
            linkTarget,
            hotelReviewsUrl,
            onClick,
        } = props;
        const {hotel, minPrice, nights, badges, hasOfferYandexPlus} =
            hotelWithMinPrice;
        const {permalink, images} = hotel;

        const convertedImages = useMemo(
            () => convertHotelOrRoomImages(images, HOTEL_IMAGE_SIZE_NAME),
            [images],
        );
        const handleMouseLeave = useCallback(() => {
            onHoverHotel?.(false);
        }, [onHoverHotel]);

        const handleMouseEnter = useCallback(() => {
            onHoverHotel?.(true);
        }, [onHoverHotel]);

        return (
            <section
                key={permalink}
                className={cx('hotelCard', className)}
                onMouseLeave={handleMouseLeave}
                onMouseEnter={handleMouseEnter}
                {...prepareQaAttributes(props)}
            >
                {images.length > 0 ? (
                    <ImagesCarousel
                        className={cx('imagesSlider')}
                        type="mini"
                        arrowsType="dark"
                        images={convertedImages}
                        hideArrows={!isHovered}
                        spaceBetween={0}
                        isDesktop
                        autoHeight
                        lazyLoad
                        isLoadImageOnViewportIntersect
                    />
                ) : (
                    <HotelImageStub
                        className={cx('emptyHotelImage')}
                        captionText={i18nBlock.noImages()}
                        illustrationProps={BUILDING_ILLUSTRATION_PROPS}
                    />
                )}
                <div
                    className={cx('content', hotelInfoClassName)}
                    onClick={onClick}
                >
                    <Link
                        url={hotelUrl}
                        className={cx('hotelLink')}
                        target={linkTarget || '_blank'}
                        rel="noopener noreferrer"
                    />
                    <div className={cx('hotelInfo')}>
                        <HotelInfoSearchCard
                            className={cx('leftContentColumn')}
                            hotelUrl={hotelUrl}
                            hotelReviewsUrl={hotelReviewsUrl}
                            hotel={hotel}
                            cardUsagePage={cardUsagePage}
                        />
                        <HotelMinPriceOffer
                            className={cx('rightContentColumn')}
                            minPrice={minPrice}
                            hotelUrl={hotelUrl}
                            nights={nights}
                            badges={badges}
                            hasOfferYandexPlus={hasOfferYandexPlus}
                            linkTarget={linkTarget}
                            {...prepareQaAttributes(props)}
                        />
                    </div>
                </div>
            </section>
        );
    };

export default React.memo(HotelCardWithMinPriceDesktop);
