import {FunctionComponent, useCallback} from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {TLinkTarget} from 'types/common/TLinkTarget';
import {IHotelOrOfferBadge} from 'types/hotels/offer/IHotelOffer';

import {getUserPlusInfo} from 'selectors/common/userInfoSelector';

import IPrice from 'utilities/currency/PriceInterface';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-HotelMinPriceOffer';
import * as i18nPlusBlock from 'i18n/components-PlusInfoPopup';
import * as i18nStaticBlock from 'i18n/hotels-seoPage-plusInfo';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Link from 'components/Link/Link';
import Price from 'components/Price/Price';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import PlusInfoPopup from 'components/PlusInfoPopup/PlusInfoPopup';
import HotelOfferBadges from 'projects/depreacted/hotels/components/HotelOfferBadges/HotelOfferBadges';

import cx from './HotelMinPriceOffer.scss';

export interface IHotelMinPriceOfferProps
    extends IWithClassName,
        IWithQaAttributes {
    minPrice: IPrice;
    hotelUrl: string;
    badges?: IHotelOrOfferBadge[];
    nights?: number;
    hasOfferYandexPlus?: boolean;
    linkTarget?: TLinkTarget;
}

const HotelMinPriceOffer: FunctionComponent<IHotelMinPriceOfferProps> =
    props => {
        const {isDesktop} = useDeviceType();
        const {
            className,
            minPrice,
            hotelUrl,
            nights,
            badges,
            hasOfferYandexPlus,
            linkTarget = isDesktop ? '_blank' : '_self',
        } = props;

        const userInfo = useUserInfo();
        const isAuth = isAuthUser(userInfo);
        const userPlusInfo = useSelector(getUserPlusInfo);
        const hasPlus = userPlusInfo.hasPlus;

        const handleClick = useCallback(e => {
            e.stopPropagation();
        }, []);

        const plusInfoPopup = (
            <PlusInfoPopup>
                <h3 className={cx('header')}>{i18nPlusBlock.header()}</h3>
                <div>
                    {isAuth && hasPlus
                        ? i18nStaticBlock.description()
                        : i18nStaticBlock.descriptionNotPlusOrAuth()}
                </div>
                <Link
                    className={cx('link')}
                    theme="normal"
                    url={'YANDEX_PLUS_URL'}
                    onClick={handleClick}
                >
                    {i18nPlusBlock.moreInfoLink()}
                </Link>
            </PlusInfoPopup>
        );

        return (
            <Box between={isDesktop ? 3 : 2} className={className}>
                {!isDesktop && (
                    <HotelOfferBadges
                        className={cx('badges')}
                        badges={badges}
                    />
                )}
                <Flex justifyContent="space-between" between="2" inline>
                    <div>
                        <Text className={cx('minPrice')} size="xl" tag="div">
                            {insertJSXIntoKey(i18nBlock.minPrice)({
                                priceNode: (
                                    <Price
                                        currency={minPrice.currency}
                                        value={minPrice.value}
                                    />
                                ),
                            })}
                        </Text>
                        <Text size="s" tag="div">
                            {!nights || nights === 1
                                ? i18nBlock.priceLabel()
                                : i18nBlock.daysPriceLabel({nights})}
                        </Text>
                    </div>
                    <ButtonLink
                        size={isDesktop ? 'm-inset' : 'l'}
                        target={linkTarget}
                        rel="noopener noreferrer"
                        theme="primary"
                        url={hotelUrl}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'buyButton',
                        })}
                    >
                        {i18nBlock.shortSelectDatesLink()}
                    </ButtonLink>
                </Flex>
                {hasOfferYandexPlus && plusInfoPopup}
                {isDesktop && (
                    <HotelOfferBadges
                        className={cx('badges')}
                        badges={badges}
                    />
                )}
            </Box>
        );
    };

export default HotelMinPriceOffer;
