import React, {useCallback} from 'react';

import {BASE_LINK_PROPS} from 'constants/linkProps';

import {IWithClassName} from 'types/withClassName';
import {IHotel, PermalinkType} from 'types/hotels/hotel/IHotel';
import {EFavoritesGoal} from 'utilities/metrika/types/goals/favorites';

import {reachGoal} from 'utilities/metrika';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {CHAR_MIDDLE_DOT, CHAR_NBSP} from 'utilities/strings/charCodes';
import getHotelNameGoalByPage from './utilities/getHotelNameGoalByPage';
import {
    IWithDataQaAttribute,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Link from 'components/Link/Link';
import Flex from 'components/Flex/Flex';
import Heading from 'components/Heading/Heading';
import Favorites from 'projects/depreacted/hotels/components/Favorites/Favorites';
import HotelGeoFeature from 'projects/depreacted/hotels/components/HotelGeoFeature/HotelGeoFeature';
import TransportAccessibility from 'projects/depreacted/hotels/components/TransportAccessibility/TransportAccessibility';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import LocationIcon from 'icons/16/Location';
import LocationOutlineIcon from 'icons/16/LocationOutline';

import HotelName from '../HotelName/HotelName';
import BudapeshtBanner from '../BudapeshtBanner/BudapeshtBanner';
import HotelGeodistance from '../HotelGeodistance/HotelGeodistance';
import HotelRatingAndCategory from '../HotelRatingAndCategory/HotelRatingAndCategory';

import cx from './HotelInfoSearchCard.scss';

/* Component Types */
export enum ECardUsagePage {
    SEARCH_PAGE = 'searchPage',
    REGION_PAGE = 'regionPage',
}

interface IHotelInfoSearchCardProps
    extends IWithClassName,
        IWithDataQaAttribute {
    cardUsagePage: ECardUsagePage;
    hotel: IHotel;
    hotelUrl: string;
    hotelReviewsUrl: string;
    onFavoriteClick?: ({
        permalink,
        isFavorite,
    }: {
        permalink: PermalinkType;
        isFavorite: boolean;
    }) => void;
}

const AMENITY_MAX_LIMIT = 10;

const HotelInfoSearchCard: React.FC<IHotelInfoSearchCardProps> = props => {
    const {
        hotel,
        hotelUrl,
        hotelReviewsUrl,
        cardUsagePage,
        className,
        onFavoriteClick,
    } = props;
    const {
        name,
        stars,
        mainAmenities,
        nearestStations,
        location,
        distanceText,
        isFavorite,
        permalink,
        isYandexHotel,
        geoFeature,
    } = hotel;

    const deviceType = useDeviceType();

    const handleHotelNameClick = useCallback(() => {
        const goal = getHotelNameGoalByPage(cardUsagePage);

        if (goal) {
            reachGoal(goal);
        }
    }, [cardUsagePage]);

    const hotelUrlProps = deviceType.isMobile
        ? {
              to: hotelUrl,
              target: '_self',
              ...BASE_LINK_PROPS,
          }
        : {
              url: hotelUrl,
              target: '_blank',
              ...BASE_LINK_PROPS,
          };

    const cityOrStopNode = location ? (
        <TextWithIcon
            spaceBetween="1"
            size={deviceType.isMobile ? 'm' : 's'}
            className={cx('location')}
            iconLeft={deviceType.isDesktop ? LocationIcon : LocationOutlineIcon}
            text={location}
        />
    ) : (
        nearestStations?.[0] && (
            <TransportAccessibility
                className={cx('transportAccessibility')}
                station={nearestStations[0]}
                textWithIconSize={deviceType.isMobile ? 'm' : 's'}
            />
        )
    );

    const handleFavoriteClick = useCallback(() => {
        if (onFavoriteClick) {
            onFavoriteClick({permalink, isFavorite});
            reachGoal(EFavoritesGoal.FAVORITES_CLICK);
        }
    }, [onFavoriteClick, permalink, isFavorite]);

    return (
        <section className={cx(className, deviceMods('hotelInfo', deviceType))}>
            <Flex justifyContent="space-between">
                <Heading level="3" className={cx('hotelName')}>
                    <Link
                        className={cx('hotelNameLink')}
                        {...hotelUrlProps}
                        onClick={handleHotelNameClick}
                    >
                        <HotelName hotelName={name} stars={stars} />
                    </Link>
                </Heading>

                {deviceType.isDesktop && onFavoriteClick && (
                    <Favorites
                        className={cx('heartIcon')}
                        onClick={handleFavoriteClick}
                        isFavorite={isFavorite}
                        deviceType={deviceType}
                    />
                )}
            </Flex>
            <HotelRatingAndCategory
                hotel={hotel}
                cardUsagePage={cardUsagePage}
                hotelReviewsUrl={hotelReviewsUrl}
                size={deviceType.isMobile ? 'm' : 's'}
            />
            <Flex
                className={cx('hotelLocationBlock')}
                between={deviceType.isDesktop ? 3 : 2}
                inline
            >
                {deviceType.isMobile && cityOrStopNode}
                {distanceText ? (
                    <HotelGeodistance
                        distanceText={distanceText}
                        isMobile={deviceType.isMobile}
                    />
                ) : (
                    <HotelGeoFeature
                        geoFeature={geoFeature}
                        isMobile={deviceType.isMobile}
                        size={deviceType.isMobile ? 'm' : 's'}
                    />
                )}
                {deviceType.isDesktop && cityOrStopNode}
            </Flex>
            {mainAmenities && mainAmenities.length > 0 && (
                <div className={cx('topFeatures')}>
                    {mainAmenities
                        .slice(0, AMENITY_MAX_LIMIT)
                        .map((amenity, index) => {
                            const notLastIndex =
                                mainAmenities.length - 1 !== index;

                            if (amenity) {
                                const {name: amenityName, id} = amenity;

                                return (
                                    <p className={cx('topFeature')} key={id}>
                                        {amenityName}
                                        <span
                                            className={cx('topFeatureSplitter')}
                                        >
                                            {notLastIndex &&
                                                `${CHAR_NBSP}${CHAR_NBSP}${CHAR_MIDDLE_DOT}`}
                                        </span>
                                    </p>
                                );
                            }

                            return null;
                        })}
                </div>
            )}
            {isYandexHotel && (
                <BudapeshtBanner
                    deviceType={deviceType}
                    {...prepareQaAttributes({parent: props, key: permalink})}
                />
            )}
        </section>
    );
};

export default HotelInfoSearchCard;
